/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/UserManager
 */
define(['log!nmodule.webEditors.rc.wb.UserManager', 'bajaux/mixin/subscriberMixIn', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/fe/registry/StationRegistry', 'nmodule/webEditors/rc/servlets/password', 'nmodule/webEditors/rc/wb/mgr/Manager', 'nmodule/webEditors/rc/wb/mgr/mgrUtils', 'nmodule/webEditors/rc/wb/mgr/commands/EditCommand', 'nmodule/webEditors/rc/wb/mgr/commands/NewCommand', 'nmodule/webEditors/rc/wb/mgr/commands/RemoveCommand', 'nmodule/webEditors/rc/wb/commands/DeleteCommand', 'nmodule/webEditors/rc/wb/mgr/commands/MgrDuplicateCommand', 'nmodule/webEditors/rc/wb/util/user/UserManagerMgrModel', 'nmodule/webEditors/rc/wb/mixin/ContextMenuSupport'], function (log, subscribable, Promise, _, DepthSubscriber, typeUtils, StationRegistry, password, Manager, mgrUtils, EditCommand, NewCommand, RemoveCommand, DeleteCommand, DuplicateCommand, UserManagerMgrModel, addContextMenuSupport) {
  'use strict';

  var getSuperTypeChain = typeUtils.getSuperTypeChain,
      getMainTableSelection = mgrUtils.getMainTableSelection,
      TYPES_TO_PRELOAD = ['baja:PasswordAuthenticator', 'baja:UserPasswordConfiguration', 'web:WebProfileConfig', 'web:MobileWebProfileConfig', 'webEditors:AuthenticationSchemeEditor', 'webEditors:ExpirationEditor', 'webEditors:RolesEditor', 'webEditors:TimeFormatEditor', 'webEditors:UserFacetsEditor', 'webEditors:UserPasswordEditor', 'webEditors:UserPrototypeEditor'],
      logError = log.severe.bind(log); //TODO: this should be more service-oriented in future.

  /**
   * As soon as the UserManager starts initializing, it will kick off a request
   * to import all the types its Edit dialog needs, then pre-populate the
   * StationRegistry with JS and agent registration info. This avoids multiple
   * hits to the registry servlets as the Edit property sheet loads, layer by
   * layer.
   *
   * @inner
   * @returns {Promise}
   */

  function preloadTypes() {
    return typeUtils.importTypes(TYPES_TO_PRELOAD).then(function (types) {
      var superTypes = _.flatten(_.map(types, getSuperTypeChain)),
          typeSpecs = _.unique(_.map(superTypes, String)),
          reg = StationRegistry.getInstance();

      return Promise.all(_.map(typeSpecs, function (typeSpec) {
        return reg.queryAll(typeSpec);
      }));
    });
  } ////////////////////////////////////////////////////////////////
  // UserManager
  ////////////////////////////////////////////////////////////////

  /**
   * User Manager.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/UserManager
   * @extends module:nmodule/webEditors/rc/wb/mgr/Manager
   * @param {Object} [params]
   */


  var UserManager = function UserManager(params) {
    var that = this;
    Manager.call(that, _.extend({
      moduleName: 'webEditors',
      keyName: 'UserManager'
    }, params));

    var subscribeDefaultPrototype = _.once(function () {
      var sub = that.$defaultPrototypeSubscriber = new DepthSubscriber(Number.MAX_VALUE);
      return sub.subscribe(that.value().get('userPrototypes').get('defaultPrototype'));
    });

    var newCommand = new NewCommand(that),
        invoke = newCommand.invoke;

    newCommand.invoke = function () {
      var args = arguments;
      return subscribeDefaultPrototype().then(function () {
        return invoke.apply(newCommand, args);
      });
    };

    that.$subscriber = new DepthSubscriber(3);
    that.getCommandGroup().add(newCommand, new EditCommand(that), new RemoveCommand(that), new DuplicateCommand(that));
    subscribable(that);
    addContextMenuSupport(that);
  };

  UserManager.prototype = Object.create(Manager.prototype);
  UserManager.prototype.constructor = UserManager;
  /**
   * Get the command to add new users.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/mgr/commands/NewCommand}
   */

  UserManager.prototype.$getNewCommand = function () {
    // noinspection JSValidateTypes
    return this.getCommandGroup().get(0);
  };
  /**
   * Get the command to edit the selected user(s).
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/mgr/commands/EditCommand}
   */


  UserManager.prototype.$getEditCommand = function () {
    // noinspection JSValidateTypes
    return this.getCommandGroup().get(1);
  };
  /**
   * Get the command to remove the selected user(s).
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/mgr/commands/RemoveCommand}
   */


  UserManager.prototype.$getRemoveCommand = function () {
    // noinspection JSValidateTypes
    return this.getCommandGroup().get(2);
  };
  /**
   * Get the command to duplicate the selected user(s).
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/commands/DuplicateCommand}
   */


  UserManager.prototype.$getDuplicateCommand = function () {
    // noinspection JSValidateTypes
    return this.getCommandGroup().get(3);
  }; ////////////////////////////////////////////////////////////
  //Context Menu Support
  ////////////////////////////////////////////////////////////

  /**
   * Trigger context menus specifically on table row right clicks
   * @override
   * @returns {string}
   */


  UserManager.prototype.getContextMenuSelector = function () {
    return 'tr';
  };
  /**
   * Callback from ContextMenuSupport to provide a chance to pre-process
   * the right click menu's CommandGroup prior to displaying it
   *
   * @param group default CommandGroup consisting of subject's menu agents
   * @returns {module:bajaux/commands/CommandGroup} updated CommandGroup
   */


  UserManager.prototype.updateMenuCommandGroup = function (group) {
    if (group) {
      group.add(this.$getDuplicateCommand(), this.$getEditCommand());
    }

    return group;
  }; /////////////////////////////////////////////////////////////////

  /**
   * When a row is double-clicked, invoke the Edit command for that row.
   * Enable the Duplicate option when one or more rows are selected
   *
   * @param {JQuery} dom
   * @returns {Promise}
   */


  UserManager.prototype.doInitialize = function (dom) {
    var that = this,
        args = arguments,
        newCmd = that.$getNewCommand();
    preloadTypes()["catch"](logError); //kick off types we know we need

    dom.on('dblclick', '.mainTable tr', function () {
      that.$getEditCommand().invoke()["catch"](logError);
    });
    return password.$isSecure().then(function (secure) {
      return newCmd.setEnabled(secure);
    }).then(function () {
      return Manager.prototype.doInitialize.apply(that, args);
    });
  };
  /**
   * Create a `UserManagerMgrModel` instance with the appropriate columns.
   *
   * @param {baja.Component} userService
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/util/user/UserManagerMgrModel>}
   */


  UserManager.prototype.makeModel = function (userService) {
    return UserManagerMgrModel.make(userService);
  };
  /**
   * Load the model into the main table and configure the commands.
   *
   * @param {baja.Component} userService - the `baja:UserService`
   * @returns {Promise}
   */


  UserManager.prototype.doLoad = function (userService) {
    var that = this,
        dupCmd = this.$getDuplicateCommand();
    dupCmd.setComponent(userService);
    return Manager.prototype.doLoad.apply(that, arguments).then(function () {
      var selection = getMainTableSelection(that);
      selection.on('changed', function () {
        that.$getEditCommand().setEnabled(!selection.isEmpty());
        password.$isSecure().then(function (secure) {
          dupCmd.setEnabled(secure && !selection.isEmpty());
        })["catch"](logError);
      });
    });
  };
  /**
   * Unsubscribe from all Components under
   * UserService -> User Prototypes -> Default Prototype on destroy.
   *
   * @returns {Promise}
   */


  UserManager.prototype.doDestroy = function () {
    var dps = this.$defaultPrototypeSubscriber;
    return Promise.all([dps && dps.unsubscribeAll(), Manager.prototype.doDestroy.apply(this, arguments)]);
  };

  return UserManager;
});
