/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/FlagConfig
 */
define(['baja!', 'lex!baja', 'bajaux/events', 'jquery', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'hbs!nmodule/webEditors/rc/wb/template/FlagConfig'], function (baja, lexs, events, $, Promise, _, BaseEditor, tplFlagConfig) {
  'use strict';

  var FLAG_CONFIG_CLASS = 'FlagConfig',
      MODIFY_EVENT = events.MODIFY_EVENT,
      LOAD_EVENT = events.LOAD_EVENT,
      DESTROY_EVENT = events.DESTROY_EVENT;
  var bajaLex = lexs[0];

  function shouldBeEnabled(symbol, isDynamic, isFrozen, isAction) {
    switch (symbol) {
      case 'r':
        return isDynamic;

      case 't':
        return false;

      case 'h':
        return true;

      case 's':
        return true;

      case 'a':
        return false;

      case 'n':
        return false;

      case 'd':
        return isFrozen;

      case 'c':
        return isAction;

      case 'o':
        return true;

      case 'x':
        return isDynamic;

      case 'f':
        return true;

      case 'A':
        return true;

      case 'p':
        return false;

      case 'R':
        return isDynamic;

      case 'm':
        return true;
      //TODO: BFlagConfig has this as "component != null". why would we edit flags without a component?

      case 'L':
        return false;

      case 'N':
        return true;

      case '1':
        return isDynamic;

      case '2':
        return isDynamic;

      case '3':
        return isDynamic;

      case '4':
        return isDynamic;
    }
  }
  /**
   * An editor for making changes to slot flags.
   *
   * Use the following `bajaux` properties to restrict which boolean checkboxes
   * are enabled/disabled:
   *
   * - `dynamic`: set to `true` to enable `readonly`, `executeOnChange`,
   *   `removeOnClone`, and all `userDefined` flags
   * - `frozen`: set to `true` to enable `defaultOnClone` flag
   * - `action`: set to `true` to enable `confirmRequired` flag
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/wb/FlagConfig
   */


  var FlagConfig = function FlagConfig() {
    BaseEditor.apply(this, arguments); //TODO: enable/disable slots depending on property/topic/action
    //TODO: this and Status editor are basically the same thing - bit string editors. can these be refactored?
    //      there are other bit string editors in Workbench too
  };

  FlagConfig.prototype = Object.create(BaseEditor.prototype);
  FlagConfig.prototype.constructor = FlagConfig;
  /**
   * Initializes Boolean editors for each possible slot flag. Also adds
   * `FlagConfig` CSS class to DOM.
   *
   * @param {JQuery} dom
   * @returns {Promise} promise to be resolved when all editors are
   * initialized
   */

  FlagConfig.prototype.doInitialize = function (dom) {
    var that = this,
        props = that.properties(),
        isDynamic = !!props.getValue('dynamic'),
        isFrozen = !!props.getValue('frozen'),
        isAction = !!props.getValue('action'),
        elems = $(tplFlagConfig({}));

    function buildEditorForFlagElement(el) {
      el = $(el);
      var symbol = String(el.data('flag')),
          flag = baja.Flags.bySymbol[symbol],
          text = bajaLex.get('flag.' + flag);
      return that.buildChildFor({
        dom: el,
        value: false,
        enabled: shouldBeEnabled(symbol, isDynamic, isFrozen, isAction),
        properties: {
          trueText: text,
          falseText: text
        },
        formFactor: 'mini'
      });
    }

    return Promise.all(_.map(elems.filter('.flag'), buildEditorForFlagElement)).then(function () {
      dom.html(elems).addClass(FLAG_CONFIG_CLASS);
      dom.on(MODIFY_EVENT, '.editor', function () {
        that.setModified(true);
        return false;
      });
      dom.on([LOAD_EVENT, DESTROY_EVENT].join(' '), '.editor', false);
    });
  };
  /**
   * Loads `true` or `false` into each Boolean editor depending on what bits
   * are set in the slot flags.
   *
   * @param {Number} value the slot flag bit mask
   * @returns {Promise} promise to be resolved when all boolean editors
   * have been loaded
   */


  FlagConfig.prototype.doLoad = function (value) {
    function loadKid(kid) {
      var symbol = String(kid.jq().data('flag')),
          flag = baja.Flags.bySymbol[symbol],
          bitSet = (flag.getMask() & value) !== 0;
      return kid.load(bitSet);
    }

    return Promise.all(this.getChildEditors().map(loadKid));
  };
  /**
   * Reads the state of each Boolean editor and assembles them into a numeric
   * bitmask representing the slot flags.
   *
   * @returns {Promise} promise to be resolved with the `Number` bitmask
   */


  FlagConfig.prototype.doRead = function () {
    var mask = 0;

    function updateMaskFromKid(kid) {
      return kid.read().then(function (val) {
        if (val) {
          var symbol = kid.jq().data('flag'),
              flag = baja.Flags.bySymbol[symbol];
          mask |= flag.getMask();
        }
      });
    } //TODO: reduceAsync?


    return Promise.all(this.getChildEditors().map(updateMaskFromKid)).then(function () {
      return mask;
    });
  };
  /**
   * Removes the `FlagConfig` CSS class from the DOM and destroys all child
   * editors.
   *
   * @returns {Promise} promise to be resolved when all child editors
   * have been destroyed
   */


  FlagConfig.prototype.doDestroy = function () {
    this.jq().removeClass(FLAG_CONFIG_CLASS);
    return this.getChildEditors().destroyAll();
  }; //TODO: enabled/readonly


  return FlagConfig;
});
