/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/ActionFirer
 */
define(['baja!', 'jquery', 'dialogs', 'Promise', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor'], function (baja, $, dialogs, Promise, fe, feDialogs, BaseEditor, ComplexCompositeEditor) {
  'use strict';

  var ACTION_FIRER_CLASS = 'ActionFirer',
      INVOKE_EVENT = 'ActionFirer:invoke',
      INVOKE_FAIL_EVENT = 'ActionFirer:invokefail'; ////////////////////////////////////////////////////////////////
  // Support editor subclasses
  ////////////////////////////////////////////////////////////////

  /**
   * Display-only editor that just shows the slot name in a button.
   *
   * @private
   * @inner
   */

  var DisplayOnlyButtonEditor = function DisplayOnlyButtonEditor() {
    BaseEditor.apply(this, arguments);
  };

  DisplayOnlyButtonEditor.prototype = Object.create(BaseEditor.prototype);
  DisplayOnlyButtonEditor.prototype.constructor = DisplayOnlyButtonEditor;

  DisplayOnlyButtonEditor.prototype.doLoad = function (value) {
    var that = this; //TODO: bad hack. nail down proper fe.makeFor behavior for an action slot. also, this might use Widget#toDisplayName instead

    var display = that.$complex.getDisplayName(that.$slot);
    that.jq().html($('<button class="ux-btn" type="button"/>').text(display));
  }; ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * An editor that will display a set of buttons, one per visible action
   * slot. Clicking the button will invoke the action using `feDialogs.action`.
   *
   * Saving the editor has no effect.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor
   * @alias module:nmodule/webEditors/rc/wb/ActionFirer
   */


  var ActionFirer = function ActionFirer() {
    ComplexCompositeEditor.apply(this, arguments);
  };

  ActionFirer.prototype = Object.create(ComplexCompositeEditor.prototype);
  ActionFirer.prototype.constructor = ActionFirer;
  /**
   * For each `Action` slot that is visible, add a clickable button for it.
   *
   * @returns {Function}
   */

  ActionFirer.prototype.getSlotFilter = function () {
    return function (slot) {
      if (slot.isAction() && !(slot.getFlags() & baja.Flags.HIDDEN)) {
        return {
          type: DisplayOnlyButtonEditor
        };
      }
    };
  };
  /**
   * Events that this editor may fire.
   */


  ActionFirer.events = {
    INVOKE_EVENT: INVOKE_EVENT,
    INVOKE_FAIL_EVENT: INVOKE_FAIL_EVENT
  };
  /**
   * Uses the `dialogs` library to show an `ActionFirer` dialog. The dialog
   * will be closed if `Cancel` is clicked or if the action completes invocation
   * (successfully or unsuccessfully).
   *
   * @param {Object} params
   * @param {baja.Component} params.component
   * @param {string} [params.title]
   * @returns {Promise} promise to be resolved with the `Action`'s return
   * value when it has finished firing. The return value will be `null` if
   * Cancel was clicked. The promise will be rejected if the `Action` firing
   * fails.
   */

  ActionFirer.showActionDialog = function (params) {
    // eslint-disable-next-line promise/avoid-new
    return new Promise(function (resolve, reject) {
      var af,
          dlg = dialogs.showCancel({
        title: params.title,
        content: '<div id="actionsDialog"/>'
      }).cancel(function () {
        return af.destroy().then(function () {
          resolve(null);
        });
      }),
          dom = $('#actionsDialog');
      params = baja.objectify(params, 'component');
      dom.one(INVOKE_EVENT + ' ' + INVOKE_FAIL_EVENT, function (e) {
        af.destroy()["finally"](function () {
          dlg.close();

          if (e.type === INVOKE_EVENT) {
            resolve();
          } else {
            reject(new Error());
          }
        });
      });
      fe.buildFor({
        dom: dom,
        value: params.component,
        type: ActionFirer
      }).then(function (e) {
        af = e;
      }, reject);
    });
  };
  /**
   * Arms an event handler that will call `feDialogs.action` whenever an
   * action button is clicked.
   *
   * After an action is invoked, it will trigger
   * `ActionFirer.events.INVOKE_EVENT` or `ActionFirer.events.INVOKE_FAIL_EVENT`
   * as appropriate.
   *
   * Also adds `ActionFirer` CSS class to the element.
   *
   * @param {JQuery} dom
   */


  ActionFirer.prototype.doInitialize = function (dom) {
    var that = this;
    dom.on('click', '.editor', function () {
      var slot = $(this).data('slot');
      feDialogs.action({
        component: that.value(),
        slot: slot
      }).then(function (result) {
        that.trigger(ActionFirer.events.INVOKE_EVENT, result);
      }, function (err) {
        that.trigger(ActionFirer.events.INVOKE_FAIL_EVENT, err);
      });
    });
    dom.addClass(ACTION_FIRER_CLASS);
    return ComplexCompositeEditor.prototype.doInitialize.apply(this, arguments);
  };
  /**
   * After destroying child editors, removes CSS classes added in
   * `initialize()`.
   *
   * @returns {Promise} promise to be resolved after CSS classes removed
   * and child editors destroyed
   */


  ActionFirer.prototype.doDestroy = function () {
    var that = this;
    return ComplexCompositeEditor.prototype.doDestroy.apply(that, arguments).then(function () {
      that.jq().removeClass(ACTION_FIRER_CLASS);
    });
  };

  return ActionFirer;
});
