/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */
define(['lex!webEditors', 'bajaux/Widget', 'Promise', 'underscore'], function (lexs, Widget, Promise, _) {
  'use strict';
  /**
   * API Status: **Private**
   * @exports nmodule/webEditors/rc/util/textUtils
   */

  var exports = {};
  var webEditorsLex = lexs[0],
      result = _.result,
      FIRST_CHAR_REGEX = /^[\w\d]+/,
      LAST_CHAR_REGEX = /[\w\d]+$/,
      BODY_REGEX = /^[\w\d-]+$/;

  function isUpperCase(c) {
    return c === c.toUpperCase();
  }
  /**
   * Return the string parameter with the first letter capitalized, and subsequent
   * upper case letters preceded by a space character. This is intended to transform
   * a programmatic name to a more user friendly style. Will turn the string 'fooBar'
   * to the string 'Foo Bar', for instance. A non-truthy parameter value will result
   * in an empty string.
   *
   * @param {String} str
   * @returns {string}
   */


  exports.toFriendly = function (str) {
    if (!str) {
      return '';
    }

    var initial = str.substring(0, 1).toUpperCase();
    return _.reduce(str.substring(1), function (memo, c) {
      return memo + (isUpperCase(c) ? ' ' + c : c);
    }, initial);
  };
  /**
   * When a user is looking at, or about to perform an operation on, an object,
   * they will be interested in seeing a display name for the object so it can
   * be made clear what they're about to change. This function attempts to
   * provide a string that, as meaningfully as possible, indicates what object
   * is being operated on.
   *
   * @param {baja.Object|baja.coll.Table|module:bajaux/Widget} obj a Component,
   * history, table, or a Widget that may have one loaded
   * @returns {Promise.<string>}
   *
   * @example
   * function confirmDelete(thingToDelete) {
   *   return textUtils.subjectToDisplay(thingToDelete)
   *     .then(function (display) {
   *       return confirm('are you sure you want to delete ' + display + '?');
   *     });
   * }
   */


  exports.subjectToDisplay = function (obj) {
    if (obj instanceof Widget) {
      return toNavDisplay(obj.value()).then(function (display) {
        return display || obj.toDisplayName();
      });
    }

    return toNavDisplay(obj).then(function (display) {
      return display || String(obj);
    });
  };

  function toNavDisplay(obj) {
    var navOrd = result(obj, 'getNavOrd');

    if (navOrd && String(navOrd.relativizeToSession()) === 'station:|slot:/') {
      return Promise.resolve(webEditorsLex.get('navTree.station'));
    }

    var navDisplayName = result(obj, 'getNavDisplayName');

    if (navDisplayName) {
      return Promise.resolve(navDisplayName);
    }

    var toConfig = result(obj, 'toConfig');

    if (toConfig) {
      return Promise.resolve(toConfig).then(function (config) {
        return config.get('navDisplayName');
      });
    }

    return Promise.resolve('');
  }
  /**
   * Return the true value if the provided hostname string is valid.
   * @param {String} address
   * @return {boolean}
   */


  exports.isHostName = function (address) {
    // Verify for null, empty and length of the address
    if (!address || address.length > 255) {
      return false;
    } // Split address into hostname labels


    var labels = address.split('.'); // For all labels

    for (var i = 0; i < labels.length; i++) {
      var currentLabel = labels[i]; // Label length should be between the length of 1 and 63 characters.

      if (!currentLabel || currentLabel.length > 63) {
        return false;
      } // First and last characters should be a-zA-Z0-9 and nothing else,
      // Characters in between can be a-zA-Z0-9 or a hyphen.


      if (!FIRST_CHAR_REGEX.test(currentLabel) || !LAST_CHAR_REGEX.test(currentLabel) || !BODY_REGEX.test(currentLabel)) {
        return false;
      }
    }

    return true;
  };

  return exports;
});
