/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*eslint-env browser */

/*jshint browser: true */
define(['jquery', 'underscore'], function ($, _) {
  'use strict';

  var DEFAULT_LONG_PRESS_DELAY = 500,
      LONG_PRESS_MOVE_TOLERANCE = 10,
      CANCELLED_BY_CONTEXT_MENU = {};
  /**
   * API Status: **Private**
   *
   * Utilities for working with HTML strings.
   *
   * @exports nmodule/webEditors/rc/util/htmlUtils
   */

  var exports = {};
  /**
   * Escape HTML characters out of a string so that it can safely be added to
   * the DOM.
   *
   * @param {String} str
   * @returns {String}
   */

  exports.escapeHtml = function (str) {
    return _.escape(str);
  };
  /**
   * @typedef {object} module:nmodule/webEditors/rc/util/htmlUtils~LongPressParams
   * @property {boolean} [mouseSupport} if true, will fire handler on a
   * long mouse press as well; otherwise touch events only.
   * @property {string} [selector] add a selector to use event delegation,
   * otherwise listen directly on element.
   * @property {number} [delay=500] amount of time necessary to keep the
   * element pressed in order to trigger contextmenu
   */

  /**
   * Define desired behavior when any element is long pressed.
   *
   * @param {JQuery} dom
   * @param {function} func this function will receive the touch event on the
   * element that was long pressed.
   * @param {module:nmodule/webEditors/rc/util/htmlUtils~LongPressParams} params
   * @returns {{ release: Function }} object with a `release` function to remove
   * associated event handlers
   */


  exports.onLongPress = function (dom, func, params) {
    var delay = params && params.delay || DEFAULT_LONG_PRESS_DELAY,
        mouseSupport = params && params.mouseSupport,
        selector = params && params.selector,
        touchstart = 'touchstart' + (mouseSupport ? ' mousedown' : ''),
        touchmove = 'touchmove' + (mouseSupport ? ' mousemove' : ''),
        touchend = 'touchend' + (mouseSupport ? ' mouseup' : ''),
        startTouch,
        ticket;

    function ontouchstart(e) {
      var touch = getSingleTouch(e);
      startTouch = touch;
      clearTimeout(ticket);

      if (touch) {
        ticket = setTimeout(function () {
          func(e);
          ticket = CANCELLED_BY_CONTEXT_MENU;
        }, delay);
      }
    }

    function ontouchmove(e) {
      var touch = getSingleTouch(e);

      if (!touch || !startTouch) {
        return;
      }

      var xDelta = Math.abs(touch.pageX - startTouch.pageX),
          yDelta = Math.abs(touch.pageY - startTouch.pageY),
          delta = Math.sqrt(xDelta * xDelta + yDelta * yDelta);

      if (delta > LONG_PRESS_MOVE_TOLERANCE) {
        clearTimeout(ticket);
      }
    }

    function ontouchend(e) {
      if (ticket === CANCELLED_BY_CONTEXT_MENU) {
        //if we already triggered the contextmenu event, consider the entire
        //touch operation "eaten" and don't propagate the touchend
        e.stopImmediatePropagation();
        return false;
      } else {
        clearTimeout(ticket);
      }
    }

    if (selector) {
      dom.on(touchstart, selector, ontouchstart).on(touchmove, selector, ontouchmove).on(touchend, selector, ontouchend);
    } else {
      dom.on(touchstart, ontouchstart).on(touchmove, ontouchmove).on(touchend, ontouchend);
    }

    return {
      release: function release() {
        dom.off(touchstart, ontouchstart).off(touchmove, ontouchmove).off(touchend, ontouchend);
      }
    };
  };
  /**
   * For environments where contextmenu does not correctly fire on a long press
   * (e.g. mobile Safari), hack it in using setTimeout.
   *
   * @param {JQuery} dom
   * @param {module:nmodule/webEditors/rc/util/htmlUtils~LongPressParams} [params]
   * @param {boolean} [params.nativeEvents] if true, fire a native click and contextmenu
   * events instead of a jquery events.
   * @param {boolean} [params.fireClick] if true, fire a native click event before
   * the contextmenu event.
   * @returns {{ release: Function }} object with a `release` function to remove
   * associated event handlers
   */


  exports.contextMenuOnLongPress = function (dom, params) {
    var nativeEvents = params && params.nativeEvents,
        fireClick = params && params.fireClick;
    return exports.onLongPress(dom, function (touch) {
      var $this = $(touch.target);

      if (nativeEvents) {
        if (fireClick) {
          var clickEventInit = {
            "button": 1,
            "clientX": touch.clientX,
            "clientY": touch.clientY,
            "screenX": touch.pageX,
            "screenY": touch.pageY
          };
          var clickEvent = new MouseEvent("click", clickEventInit);
          dom[0].dispatchEvent(clickEvent);
        }

        var mouseEventInit = {
          "button": 3,
          "clientX": touch.clientX,
          "clientY": touch.clientY,
          "screenX": touch.pageX,
          "screenY": touch.pageY
        };
        dom[0].dispatchEvent(new MouseEvent("contextmenu", mouseEventInit));
      } else {
        if (fireClick) {
          $this.trigger($.Event('click', {
            which: 1,
            pageX: touch.pageX,
            pageY: touch.pageY,
            originalEvent: touch.originalEvent
          }));
        }

        $this.trigger($.Event('contextmenu', {
          which: 3,
          pageX: touch.pageX,
          pageY: touch.pageY,
          originalEvent: touch.originalEvent
        }));
      }
    }, params);
  }; //https://stackoverflow.com/a/4358620


  function preventSelectOnShiftClick(e) {
    if (e.shiftKey && e.target !== document.activeElement) {
      var target = e.target,
          attrName = 'unselectable',
          unselectable = target.getAttribute(attrName);
      target.setAttribute(attrName, true);
      e.preventDefault();
      setTimeout(function () {
        //restore it afterwards to be safe and prevent side effects
        target.setAttribute(attrName, unselectable || false);
      }, 0);
    }
  }
  /**
   * When an element requires shift-clicking to function (for example, selecting
   * a swath of rows in a table), this can cause undesired text highlighting in
   * IE. Call this function to prevent that behavior.
   *
   * @param {JQuery} dom
   * @returns {{ release: function }} object with release function to release
   * this behavior and allow shift-click selection again
   */


  exports.preventSelectOnShiftClick = function (dom) {
    dom.on('mousedown', preventSelectOnShiftClick);
    return {
      release: function release() {
        dom.off('mousedown', preventSelectOnShiftClick);
      }
    };
  };
  /**
   * Unescape the input html string using DOMParser.
   * @param {string} stringHtml
   * @return {string} The unescaped HTML string
   */


  exports.unescapeHtml = function (stringHtml) {
    var encodedStr = stringHtml;
    var parser = new DOMParser();
    var dom = parser.parseFromString('<!doctype html><body>' + encodedStr, 'text/html');
    return dom.body.textContent;
  };

  function getSingleTouch(e) {
    var touches = e.originalEvent.touches;

    if (touches && touches.length > 1) {
      return null;
    }

    return touches ? touches[0] : e.originalEvent;
  }
  /**
   * @private
   * @function
   * @param {function} func a function to call if the user is using touch
   */


  exports.$whenTouchSupported = function () {
    var touchSupported = false,
        funcs = [],
        msToWaitForUserToTouch = 60000 * 3; //if X minutes of no touches, assume touch is not in use. we can't wait
    //forever or we'll just keep queueing functions and eat up memory.

    setTimeout(function () {
      funcs = null;
    }, msToWaitForUserToTouch);
    window.addEventListener('touchstart', function firstTouch() {
      touchSupported = true;

      if (funcs) {
        funcs.forEach(function (func) {
          func();
        });
      }

      window.removeEventListener('touchstart', firstTouch, false);
    }, false);
    return function (func) {
      return touchSupported ? func() : funcs && funcs.push(func);
    };
  }();
  /**
   * Handle focus events and call the `selectCallback` to perform the selection.
   * 'selectCallback' will not be called if editor is not enabled or readonly.
   *
   * @since Niagara 4.6
   *
   * @param {module:bajaux/Widget} widget
   * @param {string} selector - selector to filter descendant elements that trigger the focus event
   * @param {function} selectCallback - to perform selection
   * @param {object} [params]
   * @param {boolean} [params.touchOnly=true] set to false to select all on
   * focus in response to mouse events as well; by default only responds to
   * touch events
   */


  exports.selectOnFocus = function (widget, selector, selectCallback, params) {
    var dom = widget.jq(),
        touchOnly = (params && params.touchOnly) !== false;

    function armFocusHandler() {
      dom.on('focus', selector, function () {
        if (widget.isEnabled() && !widget.isReadonly()) {
          selectCallback(); //work-around for a Safari 'feature' so that we can get focus
          // after programmatic focus fails to set selection

          dom.one('mouseup', function () {
            return false;
          });
        }
      });
    }

    if (touchOnly) {
      exports.$whenTouchSupported(armFocusHandler);
    } else {
      armFocusHandler();
    }
  };
  /**
   * Sets the selectionRange on given HTML element to its entire value.
   * This is a cross browser version of select().
   *
   * @since Niagara 4.6
   *
   * @param {JQuery|HTMLElement} dom
   */


  exports.selectAll = function (dom) {
    var input = $(dom)[0];
    input.setSelectionRange(0, input.value.length);
  };

  return exports;
});
