/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/util/OrderedEventProcessor
 */
define(['Promise'], function (Promise) {
  "use strict";
  /**
   * Queues and invokes asynchronous event handlers that must be processed in order one
   * after the other. This is useful if you've got lots of incoming events where the handlers
   * are asynchronous in themselves and need to happen one after the other.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/util/OrderedEventProcessor
   */

  var OrderedEventProcessor = function OrderedEventProcessor() {
    this.$queue = [];
    this.$runningHandler = false;
  };
  /**
   * Check to see if we can process the next item in the queue. If there's nothing currently
   * running, the first item in the queue is removed and invoked.
   *
   * @private
   * @inner
   *
   * @param {module:nmodule/webEditors/rc/util/OrderedEventProcessor} processor The
   * event processor instance.
   */


  function checkQueue(processor) {
    if (!processor.$runningHandler && processor.$queue.length) {
      processor.$queue.splice(0, 1)[0]();
    }
  }
  /**
   * Wrap the event callback into a function and return it. Whenever the returned
   * function is invoked, an event handler function will be created and queued. If
   * there is nothing running in the queue, the event handler will be invoked. If not,
   * the event handler will be invoked after all other events have been processed.
   *
   * @private
   * @inner
   *
   * @param  {Function} func The function to be invoked. If the function returns
   * a promise, the next event in the queue won't be invoked until this promise
   * has been resolved.
   * @returns {Function} A function that will queue and invoke the callback. Calling
   * this function will invoke the function passed in after all prior queued event h
   * handlers have been processed.
   */


  OrderedEventProcessor.prototype.wrap = function (func) {
    var that = this;
    return function () {
      var args = arguments;
      that.$queue.push(function () {
        try {
          that.$runningHandler = true;
          Promise.resolve(func.apply(this, args))["finally"](function () {
            that.$runningHandler = false;
            checkQueue(that);
          });
        } catch (err) {
          that.$runningHandler = false;
        }
      });
      checkQueue(that);
    };
  };
  /**
   * Clear the event processor.
   */


  OrderedEventProcessor.prototype.clear = function () {
    this.$queue = [];
    this.$runningHandler = false;
  };

  return OrderedEventProcessor;
});
