/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Tony Richards
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/mixin/zoomScrollMixIn
 */
define(['bajaux/Widget', 'Promise', 'underscore'], function (Widget, Promise, _) {
  'use strict';

  var MIXIN_NAME = 'zoomScroll';
  /**
   * @callback StyleCallback A callback that returns
   * the css style to be applied.
   * @param {Number} top Pixel offset of the top of the element after scrolling / zooming has been
   *  applied.
   * @param {Number} bottom Pixel offset of the bottom of the element after scrolling / zooming has
   *  been applied.
   * @param {JQuery} element target element
   */

  /**
   * Applies the `zoomScroll` mixin to the target Widget.
   *
   * @function
   * @alias module:bajaux/mixin/zoomScrollMixIn
   * @param {module:bajaux/Widget} target The widget to apply the mixin to.
   * @param {Object} params
   * @param {Array<string>} params.zoomTargets array of css selectors that should handle pinch
   *    to zoom touch events
   * @param {Array<string>} params.moveTargets array of css selectors that should handle move
   *    touch events
   * @param {Object<String, StyleCallback>} params.applyStyles An object that maps class css
   *    selectors to a function that returns an object that can be applied as a css style.
   * @param {Number} params.zoomFactor (Optional)  a zoom factor to scale the pinch zoom in / out.
   *  default is 2.0 if not specified
   */

  var zoomScrollMixIn = function zoomScrollMixIn(target, params) {
    if (!isWidget(target)) {
      throw new Error("Zoom + Scroll mixin only applies to instances or sub-classes of Widget");
    }

    if (!_.isObject(params)) {
      throw new Error("Zoom + Scroll mixin: No zoom / scroll parameters");
    }

    if (!Array.isArray(params.zoomTargets) && !Array.isArray(params.moveTargets)) {
      throw new Error("Zoom + Scroll mixin: No Zoom / Move Targets specified");
    }

    var mixins = target.$mixins,
        doNothing = function doNothing() {
      return Promise.resolve();
    },
        doInitialize = target.doInitialize || doNothing,
        doDestroy = target.doDestroy || doNothing;

    if (!_.contains(mixins, MIXIN_NAME)) {
      mixins.push(MIXIN_NAME);
    }

    var top = 0;
    var bottom = 0;
    var lastY1 = 0;
    var lastY2 = 0;
    var zoomFactor = params.zoomFactor || 2.0;

    var handleStart = function handleStart(event) {
      if (event.touches.length === 2) {
        var topY, bottomY;

        if (event.touches[0].clientY < event.touches[1].clientY) {
          topY = 0;
          bottomY = 1;
        } else {
          topY = 1;
          bottomY = 0;
        }

        lastY1 = event.touches[topY].clientY;
        lastY2 = event.touches[bottomY].clientY;
      } else {
        if (event.touches[0]) {
          lastY1 = event.touches[0].clientY;
        }
      }
    };

    var constrain = function constrain() {
      var constrained = false; // Don't zoom out less than 100%

      if (bottom + top > 0) {
        top = bottom = 0;
        constrained = true;
      } // Don't scroll so far up that the bottom is above origin


      if (bottom > 0) {
        top += bottom;
        bottom = 0;
      } // Don't scroll so far down that the top is below origin


      if (top > 0) {
        bottom += top;
        top = 0;
      }

      return constrained;
    };

    var applyStyles = function applyStyles() {
      for (var selector in params.applyStyles) {
        if (params.applyStyles.hasOwnProperty(selector)) {
          var element = target.jq().find(selector);
          var css = params.applyStyles[selector](top, bottom, element);

          if (css) {
            element.css(css);
          }
        }
      }
    };

    var handleZoom = function handleZoom(event) {
      if (event.touches.length === 1 && event.shiftKey) {
        var curY = event.touches[0].clientY;
        var scroll = (curY - lastY1) * 2;
        lastY1 = curY;
        scroll *= 2;
        bottom -= scroll;
      } else if (event.touches.length > 1) {
        var topY, bottomY;

        if (event.touches[0].clientY < event.touches[1].clientY) {
          topY = 0;
          bottomY = 1;
        } else {
          topY = 1;
          bottomY = 0;
        }

        var Y1 = event.touches[topY].clientY;
        top += (Y1 - lastY1) * zoomFactor;
        lastY1 = Y1;
        var Y2 = event.touches[bottomY].clientY;
        bottom -= (Y2 - lastY2) * zoomFactor;
        lastY2 = Y2;
      }

      if (!constrain()) {
        event.preventDefault();
      }

      applyStyles();
    };

    var handleWheelZoom = function handleWheelZoom(event) {
      if (event.shiftKey) {
        var zoom = event.originalEvent.deltaY; // Zoom equally between top and bottom
        // TODO (or based on where the cursor is located in relation to the height of the element?)

        top -= zoom;
        bottom += zoom;
        constrain();
        applyStyles();
        event.preventDefault();
      }
    };

    var handleWheelMove = function handleWheelMove(event) {
      if (!event.shiftKey) {
        var scroll = event.originalEvent.deltaY;
        top -= scroll;
        bottom += scroll;
        constrain();
        applyStyles();
        event.preventDefault();
      }
    };

    var handleMove = function handleMove(event) {
      // Move (up / down)
      if (event.touches.length === 1 && !event.shiftKey) {
        var curY = event.touches[0].clientY;
        var scroll = (curY - lastY1) * 2;
        lastY1 = curY;
        top += scroll;
        bottom -= scroll;
        event.preventDefault();
      }

      constrain();
      applyStyles();
    };

    target.doInitialize = function (dom) {
      return Promise.resolve(doInitialize.apply(target, arguments)).then(function (result) {
        params.zoomTargets.forEach(function (selector) {
          target.jq().on('touchstart touchend', selector, handleStart);
          target.jq().on('touchmove', selector, handleZoom);
          target.jq().on('wheel', selector, handleWheelZoom);
          target.jq().on('wheel', selector, handleWheelMove);
        });
        params.moveTargets.forEach(function (selector) {
          target.jq().on('touchstart', selector, handleStart);
          target.jq().on('touchmove', selector, handleMove);
          target.jq().on('wheel', selector, handleWheelMove);
        });
        return result;
      });
    };

    target.doDestroy = function () {
      params.zoomTargets.forEach(function (selector) {
        target.jq().off('touchstart touchend', selector, handleStart);
        target.jq().off('touchmove', selector, handleZoom);
        target.jq().off('wheel', selector, handleWheelZoom);
        target.jq().off('wheel', selector, handleWheelMove);
      });
      params.moveTargets.forEach(function (selector) {
        target.jq().off('touchstart', selector, handleStart);
        target.jq().off('touchmove', selector, handleMove);
        target.jq().off('wheel', selector, handleWheelMove);
      });
      return doDestroy.apply(target, arguments);
    };
  };

  function isWidget(target) {
    return target instanceof Widget;
  }

  return zoomScrollMixIn;
});
