/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*eslint strict: "OFF" */

/**
 * API Status: **Private**
 *
 * A Handlebars block helper function for processing `EnumRange`s. The block
 * will be evaluated once per ordinal with `ordinal`, `tag`, and `display`
 * values.
 *
 * Note that if there is no lexicon specified in the `EnumRange` options, or
 * if the lexicon has not been loaded already, then `display` will fall back
 * to the raw tag value. To avoid this, ensure that you have required the
 * appropriate lexicons with `lex!` or `baja.lex()` before processing templates
 * that use this helper.
 *
 * Auto-registers as `enumRange` helper on `Handlebars` upon requiring this
 * module.
 *
 * @module nmodule/webEditors/rc/helpers/enumRangeHelper
 * @example
 *   var tpl = Handlebars.compile(
 *     '{{#enumRange enumRange}}' +
 *       'ordinal: {{ordinal}} ' +
 *       'tag: {{tag}} ' +
 *       'display: {{display}}\n' +
 *     '{{/enumRange}}'
 *   );
 *   console.log(tpl({ enumRange: baja.EnumRange.make({}) }));
 */
define(['baja!', 'lex!', 'Handlebars'], function (baja, lexPlugin, Handlebars) {
  function enumRangeHelper(enumRange, options) {
    if (!enumRange) {
      return options.inverse();
    }

    var that = this,
        fn = options.fn,
        ordinals = enumRange.getOrdinals(),
        lexiconName = enumRange.getOptions().get('lexicon'),
        ordinal,
        tag,
        display,
        lex,
        i,
        out = '',
        _ordinal = that.ordinal,
        _tag = that.tag,
        _display = that.display;

    if (lexiconName) {
      lex = lexPlugin.getLexiconFromCache(lexiconName);
    }

    for (i = 0; i < ordinals.length; i++) {
      ordinal = ordinals[i];
      tag = enumRange.getTag(ordinal);
      display = lex && lex.get({
        key: tag,
        def: tag
      });
      that.ordinal = ordinal;
      that.tag = tag;
      that.display = display || tag;
      out += fn(this);
    } //restore original state


    that.ordinal = _ordinal;
    that._tag = _tag;
    that._display = _display;
    return out;
  }

  Handlebars.registerHelper('enumRange', enumRangeHelper);
  return enumRangeHelper;
});
