/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */
define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/typeUtils'], function (baja, Promise, _, typeUtils) {
  'use strict';

  var isComponent = typeUtils.isComponent;
  /**
   * API Status: **Private**
   *
   * Utility methods for subscribing/leasing components.
   *
   * @exports nmodule/webEditors/rc/fe/baja/util/subscriberUtils
   */

  var exports = {};
  /**
   * Sometimes you may need to lease a known ORD, e.g. resolving a Service.
   * But repeated calls to `Ord#get` before the initial `get` resolves will
   * result in multiple network calls.
   *
   * This function will return the same component as long as it remains leased;
   * if it is called again after the lease expires it will automatically be
   * re-leased. Every time the function is called before the component's lease
   * expires, the lease will be extended by `leaseTime`.
   *
   * Multiple calls to this function will *not* result in multiple network
   * calls.
   *
   * @param {String|baja.Ord|baja.Component|Function} arg an ORD to resolve;
   * a Component to keep leased; or a custom function to resolve a component
   * some other way
   * @param {Object} [resolveArgs] arguments to be used to resolve the ORD.
   * @param {Number} [resolveArgs.leaseTime] the time to keep the lease before
   * it expires; if omitted, will use the default lease time configured in
   * BajaScript. This may be used both for the initial ORD resolution
   * (`{ leaseTime }`) and for the re-leasing of a `Component` (`{ time }`).
   * @returns {Function} function that returns a promise to be resolved with
   * the leased component, or rejected if the ord could not be found
   *
   * @example
   *   <caption>Create a function to lease a component by ORD</caption>
   *   var getUserService = leaseWhenNeeded('service:baja:UserService', 1000);
   *   //safe to call this function a ludicrous number of times
   *   getUserService()
   *     .then(function (userService) { });
   *
   * @example
   *   <caption>Create a function to lease a component directly</caption>
   *   baja.Ord.make('service:baja:UserService').get()
   *    .then(function (userService) {
   *      // the userService lease will expire after the timeout, but will be
   *      // renewed when getUserService is called again.
   *      var getUserService = leaseWhenNeeded(userService, 1000);
   *    });
   *
   * @example
   *   <caption>Combine a special method of resolving an ORD and keeping the
   *   component leased as needed.</caption>
   *   var getWebService = leaseWhenNeeded(function (args) {
   *         return spaceUtils.resolveService('web:WebService', args.base);
   *       }, 1000);
   *   getWebService({ base: someMountedComponent });
   */

  exports.leaseWhenNeeded = function (arg, resolveArgs) {
    var comp, getPromise, resolveOrd;

    if (typeof resolveArgs === 'number') {
      //noinspection JSValidateTypes
      resolveArgs = {
        lease: true,
        leaseTime: resolveArgs
      };
    } else if (!resolveArgs) {
      resolveArgs = {
        lease: true
      };
    }

    if (isComponent(arg)) {
      comp = arg;
      resolveOrd = _.constant(comp);
    } else if (typeof arg === 'function') {
      resolveOrd = arg;
    } else {
      resolveOrd = function resolveOrd() {
        return baja.Ord.make(arg).get(resolveArgs);
      };
    }

    function leaseComp(comp) {
      return Promise["try"](function () {
        return comp.lease({
          time: resolveArgs && resolveArgs.leaseTime
        }).then(_.constant(comp));
      });
    }

    return function (args) {
      if (comp) {
        return leaseComp(comp);
      }

      if (!getPromise) {
        getPromise = Promise.resolve(resolveOrd(args)).then(function (c) {
          comp = c;
          comp.attach('unmount', function removedHandler() {
            comp.detach('unmount', removedHandler);
            comp = null;
          });
          getPromise = null;
          return leaseComp(c);
        });
      }

      return getPromise;
    };
  };

  return exports;
});
