/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */
define(['baja!', 'baja!baja:ServiceContainer', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils'], function (baja, types, Promise, _, compUtils, typeUtils) {
  'use strict';

  var closest = compUtils.closest,
      isComplex = typeUtils.isComplex,
      isType = typeUtils.isType;
  /** @param {String} err */

  function reject(err) {
    return Promise.reject(new Error(err));
  }

  function firstKidOfType(comp, type) {
    return comp.loadSlots().then(function () {
      return comp.getSlots().properties().is(type).firstValue();
    });
  }

  function toResolveParams(resolveParams) {
    if (isComplex(resolveParams)) {
      resolveParams = {
        base: resolveParams,
        lease: true
      };
    }

    resolveParams = resolveParams || {};
    resolveParams.lease = resolveParams.lease !== false;
    return resolveParams;
  }
  /**
   * API Status: **Private**
   *
   * Utilities for locating and resolving resources from different Spaces. These
   * functions should handle unusual/unique scenarios that may be cumbersome to
   * deal with using vanilla BajaScript.
   *
   * @exports nmodule/webEditors/rc/fe/baja/util/spaceUtils
   */


  var exports = {};
  /**
   * Resolve the Station root.
   *
   * If BajaScript is online, this will resolve `station:|slot:/` as normal.
   * Otherwise, it will walk up the tree from the given `base` until it finds
   * the `baja:Station` root.
   *
   * @param {Object|baja.Complex} [resolveParams] params to be used when
   * resolving the station ORD (or, the `base` directly).
   * @param {baja.Complex} [resolveParams.base] a base Complex that lives under
   * a `baja:Station` is required to resolve the station root while BajaScript
   * is offline.
   * @param {boolean} [resolveParams.lease=true] will lease the station root by
   * default.
   * @param {number} [resolveParams.leaseTime] can specify a lease time if
   * desired.
   * @returns {Promise.<baja.Component>} promise to be resolved with the station
   * root
   */

  exports.resolveStation = function (resolveParams) {
    resolveParams = toResolveParams(resolveParams);

    if (!baja.isOffline()) {
      return baja.Ord.make('station:|slot:/').get(resolveParams);
    }

    var base = resolveParams && resolveParams.base;

    if (!isComplex(base)) {
      return reject('base required when BajaScript is offline');
    }

    var station = closest(base, 'baja:Station');

    if (!station) {
      return reject('base is not a descendant of a baja:Station');
    }

    return station.lease().then(_.constant(station));
  };
  /**
   * Resolve a Service of a particular Type.
   *
   * If BajaScript is online, this will resolve a `service:` ORD as normal.
   * Otherwise, it will walk up the tree from the given `base` and check the
   * `Station`'s `Services` container manually.
   *
   * @param {String|Type} serviceType the type of the Service to resolve
   * @param {Object|baja.Complex} [resolveParams] params to be used when
   * resolving the service ORD (or, the `base` directly).
   * @param {baja.Complex} [resolveParams.base] a base Complex that lives under
   * a `baja:Station` is required to resolve a service while BajaScript is
   * offline.
   * @param {boolean} [resolveParams.lease=true] will lease the resolved service
   * by default.
   * @param {number} [resolveParams.leaseTime] can specify a lease time if
   * desired.
   * @returns {Promise.<baja.Component>} promise to be resolved with the Service
   * instance
   */


  exports.resolveService = function (serviceType, resolveParams) {
    if (!isType(serviceType)) {
      return reject('type required');
    }

    resolveParams = toResolveParams(resolveParams);

    if (!baja.isOffline()) {
      return baja.Ord.make('service:' + serviceType).get(resolveParams);
    }

    return exports.resolveStation(resolveParams).then(function (station) {
      //since N4, Services is a frozen slot.
      return firstKidOfType(station.get('Services'), serviceType);
    }).then(function (service) {
      var lease = resolveParams.lease,
          time = resolveParams.leaseTime;

      if (lease) {
        return service.lease({
          time: time
        }).then(_.constant(service));
      }

      return service;
    });
  };

  return exports;
});
