/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */
define(['baja!'], function (baja) {
  'use strict';
  /**
   * API Status: **Private**
   *
   * Utilities for Baja Slots.
   *
   * @exports nmodule/webEditors/rc/fe/baja/util/slotUtils
   */

  var exports = {};
  var ACTION_ICON = ['/module/icons/x16/action.png'],
      TOPIC_ICON = ['/module/icons/x16/topic.png'],
      SLOT_CLASS_PREFIX = 'slot-';
  /**
   * Certain slots should not be shown to any UI widgets; use this method
   * to determine if a certain slot should be hidden from UI elements. These
   * slots include:
   *
   * - slots with the `HIDDEN` slot flag,
   * - `wsAnnotation` slots,
   * - `displayNames` slots,
   * - `baja:Link` slots
   *
   * @param slot
   * @returns {boolean}
   */

  exports.isUiVisible = function (slot) {
    if (slot.getFlags() & baja.Flags.HIDDEN) {
      return false;
    }

    if (slot.isProperty() && (slot.getName() === 'wsAnnotation' || slot.getType().is('baja:Link'))) {
      return false;
    } //action or topic


    return true;
  };
  /**
   * Retrieve the icon for the given slot.
   *
   * @param {baja.Slot} slot
   * @returns {Array.<String>} an array of image URIs
   */


  exports.getSlotIcon = function (slot) {
    return slot.isProperty() ? slot.getType().getIcon().getImageUris() : slot.isAction() ? ACTION_ICON : TOPIC_ICON;
  };
  /**
   * Get the display value for a Slot's Type. For an Action, shows the slot's
   * param and return types; for a Topic shows the event type.
   *
   * @param {baja.Slot} slot
   * @returns {String}
   */


  exports.getTypeDisplay = function (slot) {
    if (!(slot instanceof baja.Slot)) {
      throw new Error('baja.Slot required');
    }

    if (slot.isAction()) {
      return (slot.getReturnType() || 'void') + ' (' + (slot.getParamType() || 'void') + ')';
    }

    if (slot.isTopic()) {
      //TODO: getEventType() not in bajaScript docs
      //TODO: baja.$ not documented
      return String(slot.getEventType());
    }

    return String(slot.getType());
  }; //TODO: this should be in BajaScript.

  /**
   * Convert slot flags to a string in the same way Workbench does.
   *
   * @param {Number|baja.Slot} flags
   * @returns {string}
   */


  exports.getFlagsDisplay = function (flags) {
    if (flags instanceof baja.Slot) {
      flags = flags.getFlags();
    }

    var symbols = [],
        allFlags = baja.Flags.flags,
        flag,
        i;

    for (i = 0; i < allFlags.length; i++) {
      flag = allFlags[i];

      if (flag.getMask() & flags) {
        symbols.push(flag.getSymbol());
      }
    }

    return symbols.join('');
  };
  /**
   * Converts a slot name into a valid CSS class name.
   *
   * @param {String|baja.Slot} slot a slot or an unescaped slot name
   * @returns {String} a valid CSS class name
   */


  exports.toCssClass = function (slot) {
    return SLOT_CLASS_PREFIX + String(slot).replace(/\$/g, '_');
  };
  /**
   * Converts the CSS class name from `toCssClass` back into an unescaped
   * slot name.
   *
   * @param {String} cssClass the CSS class generated from `toCssClass`
   * @returns {String} an unescaped slot name
   */


  exports.fromCssClass = function (cssClass) {
    return String(cssClass).replace(/^slot-/, '').replace(/_/g, '$');
  };
  /**
   * @param {string} path path of keys e.g. `a/b/c`
   * @param {string} pattern query pattern, e.g. `a/&#40;/c`
   * @returns {boolean} true if the path matches the pattern
   */


  exports.globMatch = function (path, pattern) {
    if (path === pattern) {
      return true;
    }

    var pathMembers = path.split('/'),
        patternMembers = pattern.split('/');
    var pathIndex = 0;

    for (var i = 0, len = patternMembers.length; i < len; ++i, ++pathIndex) {
      var patternMember = patternMembers[i],
          pathMember = pathMembers[pathIndex];

      switch (patternMember) {
        case '*':
          break;
        //match any

        case '**':
          var nextPatternMember = patternMembers[++i];

          if (!nextPatternMember) {
            return true; //ending in globstar is a match
          } //advance to next match


          if (nextPatternMember !== '*') {
            var nextPatternMemberRegex = new RegExp('^' + nextPatternMember + '$');

            while (pathMember && !pathMember.match(nextPatternMemberRegex)) {
              pathMember = pathMembers[++pathIndex];
            }

            if (!pathMember) {
              return false; //no match found
            }
          }

          if (nextPatternMember === '*' && i === patternMembers.length - 1) {
            return true;
          } //???


          break;

        default:
          if (!pathMember || !pathMember.match(new RegExp('^' + patternMember + '$'))) {
            return false;
          }

      }
    } //true if we hit every member in the path without failures


    return pathIndex >= pathMembers.length;
  };

  return exports;
});
