/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */
define(['baja!', 'Promise', 'underscore', 'nmodule/js/rc/asyncUtils/asyncUtils', 'nmodule/webEditors/rc/fe/baja/util/ComplexDiff', 'nmodule/webEditors/rc/fe/baja/util/typeUtils'], function (baja, Promise, _, asyncUtils, ComplexDiff, typeUtils) {
  'use strict';

  var doRequire = asyncUtils.doRequire;
  var getUniqueSlotName = typeUtils.getUniqueSlotName,
      isComplex = typeUtils.isComplex,
      isComponent = typeUtils.isComponent,
      isSimple = typeUtils.isSimple,
      isType = typeUtils.isType,
      isValue = typeUtils.isValue;
  var ADD_SLOT_EDITOR_RJS = 'nmodule/webEditors/rc/wb/commands/AddSlotEditor'; //avoid circular dependency

  var getFeDialogs = _.once(function () {
    return doRequire('nmodule/webEditors/rc/fe/feDialogs');
  }); ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /** @param {String} err */


  function reject(err) {
    return Promise.reject(new Error(err));
  }

  function toNavOrd(value) {
    return value.getNavOrd();
  }
  /**
   * Check to see if the value is mounted and has a nav ORD. If so, it's a
   * candidate for doing a station-side transfer.
   *
   * @inner
   * @param {*} value
   * @returns {boolean}
   */


  function hasNavOrd(value) {
    return !!(baja.hasType(value, 'baja:INavNode') && value.getNavOrd());
  }
  /**
   * Make sure we have a component, an array of values to add to it, and either
   * they're all mounted or all unmounted.
   *
   * @inner
   * @param {baja.Component} comp
   * @param {Array.<baja.Value>} values
   * @throws {Error}
   */


  function validateAddParams(comp, values) {
    if (!isComponent(comp)) {
      throw new Error('baja:Complex required');
    } //TODO: again, need isArrayOf


    if (!Array.isArray(values)) {
      throw new Error('values array required');
    }

    var b = hasNavOrd(values[0]);

    for (var i = 1; i < values.length; i++) {
      if (hasNavOrd(values[i]) !== b) {
        throw new Error('values must be all mounted or all unmounted');
      }
    }
  }
  /**
   * Add the values to the target component stationside using the Transfer API.
   *
   * @inner
   * @param {baja.Component} target
   * @param {Array.<baja.NavNode>} values
   * @param {Array.<String>} [names]
   * @returns {Promise}
   */


  function addMounted(target, values, names) {
    var sourceOrds = _.map(values, toNavOrd);

    var obj = {
      sourceOrds: sourceOrds,
      target: target
    };

    if (names) {
      obj.names = names;
    }

    return baja.transfer.copy(obj);
  }
  /**
   * Add the values to the target browserside using `Component#add()`.
   *
   * @inner
   * @param {baja.Component} target
   * @param {Array.<baja.NavNode>} values
   * @param {Array.<String>} [names]
   * @param {Array.<Number>} [flags]
   * @returns {Promise}
   */


  function addUnmounted(target, values, names, flags) {
    var batch = new baja.comm.Batch();
    return batch.commit(Promise.all(values.map(function (value, i) {
      return target.add({
        slot: names && names[i] || getUniqueSlotName(target, value.getType().getTypeSpec()) + '?',
        value: value.newCopy(),
        flags: flags[i],
        batch: batch
      });
    })));
  }

  function doAdd(target, values, names, flags) {
    return (hasNavOrd(values[0]) ? addMounted : addUnmounted)(target, values, names, flags || []);
  }
  /**
   * Resolves a constructor that can be used to retrieve "add slot" details
   * from the user. We have to asynchronously require the module because
   * it extends `PropertySheet` - and without the async trickery we have a
   * circular dependency: `PropertySheet` -> `AddSlotCommand` -> `PropertySheet`.
   *
   * @inner
   * @function
   * @returns {Promise}
   */


  function getAddSlotEditor() {
    return doRequire(ADD_SLOT_EDITOR_RJS);
  } ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * API Status: **Private**
   *
   * Utility functions for working with Complexes and Components.
   *
   * @exports nmodule/webEditors/rc/fe/baja/util/compUtils
   */


  var exports = {};
  /**
   * Shows a dialog to the user asking for a slot name (and, unless specified,
   * type spec and slot flags). The resolved object can be passed directly as
   * a parameter to `Component#add` to complete the add slot operation.
   *
   * @param {baja.Component} value the component to which we wish to add a new
   * slot. Note that simply calling `promptForAddParameters` will not actually
   * add a slot - `value#add()` must still be called.
   *
   * @param {Object} [params]
   * @param {String} [params.display] if `simple`, will only show the slot
   * name editor to the user - no type spec or flags.
   * @param {String} [params.name] allow a predetermined slot name. If
   * omitted, a default name will be derived from the type spec.
   * @param {String} [params.typeSpec] allow a predetermined type spec.
   * *Required* if `display` is `simple`.
   * @returns {Promise} promise to be resolved with an object that can
   * be passed to `value#add` to complete the add slot operation, or with `null`
   * if the user cancels the dialog.
   */

  exports.promptForAddParameters = function (value, params) {
    return getAddSlotEditor().then(function (AddSlotEditor) {
      return AddSlotEditor.doDialog(params, value);
    });
  };
  /**
   * Add to the given component new copies of all values passed in. Slot names
   * will be auto-generated based on the type spec of each value if not
   * specified.
   *
   * If the values passed in are valid nav nodes, the add operation will
   * be done station-side using the Transfer API. If the values passed in are
   * not nav nodes, the add will be done locally using `Component#add` and
   * `newCopy()`. If a mix of mounted and unmounted values are given, the
   * operation will fail and no copying will be done.
   *
   * @param {baja.Component} comp
   * @param {Array.<baja.Value>} values
   * @param {Object} [params]
   * @param {Array.<String>} [params.names] if given, can specify slot names
   * for the added values. If not specified, slot names will be generated
   * from the value type specs.
   *
   * @returns {Promise} promise to be resolved when new copies of all
   * values have been added to the target component. It will be resolved
   * with an array of the Slots that were added to the component.
   */


  exports.bulkAdd = function (comp, values, params) {
    return Promise["try"](function () {
      validateAddParams(comp, values);

      if (!values.length) {
        return;
      }

      var names = params && params.names; //when adding more than one value, don't prompt for a slot name.

      if (values.length > 1) {
        return doAdd(comp, values, names);
      }

      var value = values[0];
      var slotName = names && names[0] || value.getNavName() || value.getType().getTypeName();
      return getFeDialogs().then(function (feDialogs) {
        return feDialogs.showFor({
          value: baja.SlotPath.unescape(comp.getUniqueName(slotName)),
          formFactor: 'mini'
        });
      }).then(function (slotName) {
        if (slotName) {
          return doAdd(comp, values, [baja.SlotPath.escape(slotName)]);
        }
      });
    });
  };
  /**
   * Copy a set of properties from one Complex onto another.
   *
   * @param {baja.Complex} source
   * @param {baja.Complex} target
   * @param {Array.<String|baja.Slot>} props array of Simple properties to copy.
   * If any of these are Complexes, the copy will fail (Complexes will not be
   * re-parented).
   * @param {baja.comm.Batch} [batch] batch to use for the target `set`
   * operations. If no batch is provided, a new one will be created and used
   * internally. (The batch will be added to the target `set`s synchronously,
   * so safe to use with `batchSaveMixin`.)
   * @returns {Promise} promise to be resolved when all properties are
   * copied, or rejected if the copy fails for any reason
   */


  exports.bulkCopy = function (source, target, props, batch) {
    if (!isComplex(source) || !isComplex(target)) {
      return reject('source and target must be Complex');
    }

    if (!Array.isArray(props)) {
      return reject('props array required');
    }

    for (var i = 0, prop; i < props.length; i++) {
      prop = props[i];

      if (!source.has(prop) || !isComponent(target) && !target.has(prop)) {
        return reject('missing property ' + prop);
      }

      if (!target.getSlot(prop).getType().is(source.getSlot(prop).getType())) {
        return reject('different types for property ' + prop);
      }
    }

    var batchParam = batch;
    var batchToUse = batchParam || new baja.comm.Batch();
    var sets = props.map(function (prop) {
      return target.set({
        slot: prop,
        value: source.get(prop),
        batch: batchToUse
      });
    });

    if (!batchParam) {
      batchToUse.commit();
    }

    return Promise.all(sets);
  };
  /**
   * Check to see if I have permissions to make changes to the specified slot
   * on the Complex.
   *
   * @param {baja.Complex} comp
   * @param {baja.Property|String} slot
   * @returns {boolean}
   */


  exports.canWriteSlot = function (comp, slot) {
    if (!isComplex(comp)) {
      throw new Error('complex required');
    }

    if (!slot) {
      throw new Error('slot required');
    }

    if (!comp.has(slot)) {
      return false;
    }

    if (comp.getFlags(slot) & baja.Flags.READONLY) {
      return false;
    }

    while (isComplex(comp) && !isComponent(comp)) {
      slot = comp.getPropertyInParent();
      comp = comp.getParent();
    }

    if (!isComponent(comp)) {
      return true;
    }

    if (comp.getFlags(slot) & baja.Flags.READONLY) {
      return false;
    }

    var permissions = comp.getPermissions();

    if (permissions.hasAdminWrite()) {
      return true;
    } else if (permissions.hasOperatorWrite() && comp.getFlags(slot) & baja.Flags.OPERATOR) {
      return true;
    }

    return false;
  };
  /**
   * Write changes to a slot on a Complex. This can be handled in a number of
   * different ways:
   *
   * - If the slot does not exist, it will be added.
   * - If the value is a Simple, if will be directly set.
   * - If the value is a Complex of the same Type as the existing value, a diff
   *   will be applied in order to write the changes while still respecting
   *   edit-by-ref semantics.
   * - If the value is a Complex of a different Type, the value will be
   *   overwritten with a new copy.
   * - If the value is a ComplexDiff, it will be applied to the value in the
   *   slot (which must itself be a Complex).
   *
   * In essence, you provide a value describing "what I want this slot on this
   * complex to look like," and this function handles the details.
   *
   * Be careful when writing Complexes of an unknown depth. This function will
   * not perform any subscription and you may lose data. See
   * {@link module:nmodule/webEditors/rc/fe/baja/util/DepthSubscriber} if you
   * need to ensure any component trees are fully subscribed before calling
   * this function.
   *
   * @param {baja.Complex} comp
   * @param {baja.Property|String} slot
   * @param {baja.Value|module:nmodule/webEditors/rc/fe/baja/util/ComplexDiff|baja.Value} value
   * @param {object} [params]
   * @param {baja.comm.Batch} [params.batch]
   * @returns {Promise} to be resolved when the slot has been written
   */


  exports.writeSlot = function (comp, slot, value, params) {
    var kid = comp.get(slot);

    if (value instanceof ComplexDiff) {
      return value.apply(kid, params);
    } else if (isValue(value)) {
      if (kid === null || isSimple(value) || !kid.getType().equals(value.getType())) {
        return exports.putSlot(comp, slot, value.newCopy(true), params);
      } else {
        return ComplexDiff.diff(kid, value).apply(kid, params);
      }
    } else {
      return reject('value must be a ComplexDiff or baja:Value');
    }
  };
  /**
   * If the slot exists, set its value; otherwise add a new slot.
   *
   * Be careful if the complex is mounted but not leased - if slots are not
   * loaded, it may do an add when a set was intended.
   *
   * @param {baja.Complex} comp
   * @param {baja.Slot|String} slot
   * @param {baja.Value} value
   * @param {object|baja.comm.Batch} [params]
   * @param {baja.comm.Batch} [params.batch]
   * @returns {Promise}
   */


  exports.putSlot = function (comp, slot, value, params) {
    var reserve = baja.comm.Batch.reserve(params);
    return reserve.commit(comp[comp.has(slot) ? 'set' : 'add']({
      slot: String(slot),
      value: value,
      batch: reserve
    }));
  };
  /**
   * Convert a Complex into an object literal, where each Property corresponds
   * to one property on the object.
   *
   * @param {baja.Complex} comp
   * @param {object} [params]
   * @param {function} [params.filter] optional slot filter
   * @param {boolean} [params.deep] set it to true if child Complexes should also
   * be converted to object literals; otherwise will be left as Complexes
   * @returns {object}
   */


  exports.toObject = function (comp, params) {
    var conversion = {};

    var filter = params && params.filter || _.constant(true);

    var deep = params && params.deep;
    comp.getSlots().properties().filter(filter).each(function (slot) {
      if (deep && isComplex(comp.get(slot))) {
        conversion[slot] = exports.toObject(comp.get(slot), params);
      } else {
        conversion[slot] = comp.get(slot);
      }
    });
    return conversion;
  }; // TODO: This functionality should be moved into the regular newCopy.

  /**
   * Generates a copy with display names intact.
   * @param {*} value
   * @returns {*} a new copy of a baja Value; if not a baja Value it will be
   * returned directly.
   */


  exports.newCopyComplete = function (value) {
    if (!baja.hasType(value)) {
      return value;
    }

    var newComponent = value.newCopy(true);

    if (isComplex(newComponent)) {
      var display = value.getDisplay();
      var getDisplay = newComponent.getDisplay;

      newComponent.getDisplay = function () {
        var args = arguments;
        return args.length ? getDisplay.apply(this, args) : display;
      };

      value.getSlots().properties().toArray().forEach(function (slot) {
        newComponent.set({
          slot: slot,
          value: exports.newCopyComplete(value.get(slot)),
          cx: {
            display: value.getDisplay(slot),
            displayName: value.getDisplayName(slot)
          }
        });
      });
    }

    return newComponent;
  };
  /**
   * Find the closest ancestor of the given Complex, including the Complex
   * itself, that has the given Type.
    * @param {baja.Complex} comp
   * @param {String|Type} type
   * @returns {baja.Complex|null} the closest ancestor (or the given Complex
   * itself) that has the given Type, or null if not found.
   */


  exports.closest = function (comp, type) {
    if (!isComplex(comp)) {
      throw new Error('complex required');
    }

    if (!isType(type)) {
      throw new Error('type required');
    }

    while (comp && !baja.hasType(comp, type)) {
      comp = comp.getParent();
    }

    return comp;
  };
  /**
   * Given two Complex instances where one is the ancestor (or the same
   * instance) of another instance, find the array of slots leading to the
   * descendant Complex.
   *
   * @param {baja.Complex} ancestor
   * @param {baja.Complex} descendant
   * @returns {Array.<baja.Property>} array of property slots leading to the
   * descendant. Will be length 0 if ancestor and descendant are the same
   * instance.
   * @throws {Error} if ancestor or descendant are not valid Complexes, or if
   * the descendant is not actually in the ancestor's tree
   */


  exports.slotPathFromAncestor = function (ancestor, descendant) {
    if (!isComplex(ancestor)) {
      throw new Error('ancestor required');
    }

    if (!isComplex(descendant)) {
      throw new Error('descendant required');
    }

    var slots = [];
    var c = descendant;

    while (c && c !== ancestor) {
      slots.push(c.getPropertyInParent());
      c = c.getParent();
    }

    if (c !== ancestor) {
      throw new Error('descendant was not in ancestor\'s component tree');
    }

    return slots.reverse();
  };
  /**
   * If `complex` is a Component, will simply return the slot facets. If
   * `complex` is a Struct, will merge in slot facets from the nearest ancestor
   * Component (if present).
   * @param {baja.Complex} complex
   * @param {baja.Slot|string} slot
   * @returns {baja.Facets}
   */


  exports.getMergedSlotFacets = function (complex, slot) {
    var comp = exports.closest(complex, 'baja:Component');

    if (comp) {
      var propertyPath = exports.slotPathFromAncestor(comp, complex);
      return baja.Facets.mergeSlotFacets(comp, propertyPath.concat(slot));
    } else {
      return complex.getFacets(slot);
    }
  };
  /**
   * Get a nav ord for any Complex or Table.
   *
   * If the Complex is a Component, return that component's Nav ORD directly.
   * Otherwise, find the Struct's parent Component and return a new ORD based
   * off the parent Component's Nav ORD that will resolve to that Struct.
   *
   * If there is no parent Component or if it is not mounted, return null.
   *
   * For a Table, resolve the `navOrd` from the Table config.
   *
   * @param {baja.Complex|baja.NavNode|baja.Ord|baja.coll.Table} object
   * @returns {baja.Ord|null}
   */


  exports.getNavOrd = function (object) {
    var config = _.result(object, 'toConfig');

    if (config) {
      return Promise.resolve(config).then(function (config) {
        return config.get('navOrd');
      });
    }

    if (baja.hasType(object, 'baja:Ord')) {
      return Promise.resolve(object);
    }

    var navNode = getNavNode(object);
    var navOrd = navNode && navNode.getNavOrd();

    if (!navOrd || navNode === object) {
      return Promise.resolve(navOrd);
    }

    return Promise.resolve(baja.Ord.make({
      base: navOrd,
      child: 'slot:' + exports.slotPathFromAncestor(navNode, object).join('/')
    }));
  };

  function getNavNode(object) {
    if (baja.hasType(object, 'baja:INavNode')) {
      return object;
    }

    if (isComplex(object)) {
      return exports.closest(object, 'baja:INavNode');
    }
  } //TODO: move to BajaScript Complex.js?

  /**
   * Get the parent component for the given Complex. If the Complex itself
   * is a Component, return it directly; otherwise walk up the parent list until
   * a Component is found.
   *
   * For instance, this method would be used to check write permissions on a
   * Struct by retrieving them from its parent Component (since a Struct has no
   * permissions of its own).
   *
   * (Equivalent to `getClosest(complex, 'baja:Component')`).
   *
   * @param {baja.Complex|*} comp the Complex to retrieve a parent component
   * for
   * @returns {baja.Component|null} the parent Component, or `null` if not found
   */


  exports.getParentComponent = function (comp) {
    //noinspection JSValidateTypes
    return exports.closest(comp, 'baja:Component');
  };
  /**
   * Check to see if one Complex is an ancestor of another.
   * @param {baja.Complex} ancestor
   * @param {baja.Complex} descendant
   * @returns {boolean} true if the first Complex is an ancestor of the second
   * (but not the same instance)
   */


  exports.isAncestorOf = function (ancestor, descendant) {
    if (!isComplex(ancestor)) {
      throw new Error('ancestor required');
    }

    if (!isComplex(descendant)) {
      throw new Error('descendant required');
    }

    var comp = descendant;

    while (comp = comp.getParent()) {
      if (comp === ancestor) {
        return true;
      }
    }

    return false;
  };

  return exports;
});
