function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

function _defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } }

function _createClass(Constructor, protoProps, staticProps) { if (protoProps) _defineProperties(Constructor.prototype, protoProps); if (staticProps) _defineProperties(Constructor, staticProps); return Constructor; }

/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/util/Attachable
 */
define(['underscore', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/fe/baja/util/slotUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils'], function (_, compUtils, slotUtils, typeUtils) {
  'use strict';

  var isAncestorOf = compUtils.isAncestorOf,
      slotPathFromAncestor = compUtils.slotPathFromAncestor;
  var globMatch = slotUtils.globMatch;
  var isComponent = typeUtils.isComponent;
  var each = _.each,
      flatten = _.flatten;
  /**
   * Utility class for managing Baja event subscription and unsubscription for
   * a component and its children.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/fe/baja/util/Attachable
   * @param {baja.Component} comp the component onto which to attach event
   * handlers
   */

  var Attachable = /*#__PURE__*/function () {
    function Attachable(comp) {
      _classCallCheck(this, Attachable);

      if (!isComponent(comp)) {
        throw new Error('component required');
      }

      var that = this;
      that.$handlers = {};
      that.$comp = comp;

      var addedHandler = that.$addedHandler = function (prop) {
        var newComp = this.get(prop);

        if (isComponent(newComp)) {
          each(that.$handlers, function (handlerArray, eventPath) {
            if (!handlerArray) {
              return;
            }

            var targets = that.$getEventTargets(eventPath, comp);
            targets.forEach(function (_ref) {
              var component = _ref.component,
                  event = _ref.event;

              if (component === newComp || isAncestorOf(newComp, component)) {
                handlerArray.forEach(function (handler) {
                  component.attach(event, handler);
                  component.attach('added', addedHandler);
                });
              }
            });
          });
        }
      };

      comp.attach('added', addedHandler);
    }
    /**
     * Find all nested targets for this Baja event. Supports regex syntax.
     *
     * @private
     * @param {String} eventString Baja event name, or slot path-like syntax
     * @param {baja.Component} component the root component of the Attachable
     * @returns {Array.<{ component: baja.Component, event: string }>}
     */


    _createClass(Attachable, [{
      key: "$getEventTargets",
      value: function $getEventTargets(eventString, component) {
        var _this = this;

        if (eventString.match(/\*\*/)) {
          return getEventTargetsRecursive(eventString, component);
        }

        var slotPatterns = eventString.split('/');
        var event = String(slotPatterns.pop());
        var descendants = slotPatterns.slice(1).concat([event]).join('/');

        if (slotPatterns.length) {
          var regex = new RegExp('^' + slotPatterns[0]);
          var kids = component.getSlots().is('baja:Component').filter(function (slot) {
            return slot.getName().match(regex);
          }).toValueArray();
          var kidTargets = kids.map(function (kid) {
            return _this.$getEventTargets(descendants, kid);
          });
          return flatten(kidTargets);
        }

        return [{
          component: component,
          event: event
        }];
      }
      /**
       * Attach event handlers for a specified set of Baja events.
       *
       * Event keys can be straight Baja event names, in which they will be attached
       * to the main component passed to the constructor. They can also attach
       * handlers to child components using a slot path-like syntax; see examples.
       *
       * When attaching to child components, remember that no Baja event handlers
       * will be fired unless those children are subscribed. Either subscribe them
       * manually or use a `DepthSubscriber`.
       *
       * @param {Object} obj a map from event strings to handler functions
       *
       * @example
       *   <caption>Attach handlers directly to a component.</caption>
       *
       *   resolveAndSubscribe('station:|slot:/someComponent')
       *     .then(function (comp) {
       *       var att = new Attachable(comp);
       *       att.attach({
       *         added: function (prop) {
       *           baja.outln(prop + ' added');
       *         },
       *         removed: function (prop) {
       *           baja.outln(prop + ' removed');
       *         }
       *       });
       *
       *       //then later...
       *
       *       att.detach();
       *     });
       *
       * @example
       *   <caption>Attach handlers to a component and also to specified child
       *   components.</caption>
       *
       *   resolveAndSubscribeDeep('station:|slot:/Services', 2)
       *     .then(function (services) {
       *       var att = new Attachable(services);
       *       att.attach({
       *         added: function (prop) {
       *           baja.outln(prop + ' service added');
       *         },
       *         'UserService/added': function (prop) {
       *           baja.outln(prop + ' user added');
       *         }
       *       });
       *     });
       *
       * @example
       *   <caption>Attach handlers to multiple nested children of the component,
       *   using a regex syntax.</caption>
       *
       *   resolveAndSubscribeDeep('station:|slot:/Services/UserService', 2)
       *     .then(function (userService) {
       *       var att = new Attachable(userService);
       *       att.attach({
       *         '.+/changed': function (prop) {
       *           if (prop.getName() === 'fullName') {
       *             baja.outln('user ' + this.getName() + ' changed full name');
       *           }
       *         }
       *       });
       *     });
       */

    }, {
      key: "attach",
      value: function attach(obj) {
        var _this2 = this;

        var comp = this.$comp;
        var handlers = this.$handlers;
        var addedHandler = this.$addedHandler;
        each(obj, function (handler, event) {
          var arr = handlers[event] || (handlers[event] = []);

          var targets = _this2.$getEventTargets(event, comp);

          arr.push(handler);
          targets.forEach(function (_ref2) {
            var component = _ref2.component,
                event = _ref2.event;
            component.attach(event, handler);
            component.attach('added', addedHandler);
          });
        });
      }
    }, {
      key: "detach",

      /**
       * Detach all Baja event handlers previously attached using `attach()`.
       */
      value: function detach() {
        var _this3 = this;

        var comp = this.$comp;
        var handlers = this.$handlers;
        var addedHandler = this.$addedHandler;
        comp.detach('added', addedHandler);
        each(handlers, function (arr, event) {
          var targets = _this3.$getEventTargets(event, comp);

          arr.forEach(function (handler) {
            targets.forEach(function (_ref3) {
              var component = _ref3.component,
                  event = _ref3.event;
              component.detach(event, handler);
              component.detach('added', addedHandler);
            });
          });
        });
      }
    }]);

    return Attachable;
  }();

  function getEventTargetsRecursive(event, comp) {
    var slotPatterns = event.split('/');
    var eventName = slotPatterns.pop();
    var slotPattern = slotPatterns.join('/');
    return flatten(getAllKids(comp)).filter(function (kid) {
      return globMatch(slotPathFromAncestor(comp, kid).join('/'), slotPattern);
    }).map(function (component) {
      return {
        component: component,
        event: eventName
      };
    });
  }

  function getAllKids(comp) {
    return comp.getSlots().is('baja:Component').toValueArray().map(function (kid) {
      return [kid].concat(getAllKids(kid));
    });
  }

  return Attachable;
});
