/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/compat/PikadayDateEditor
 */
define(['baja!', 'baja!baja:Month,baja:Weekday', 'bajaScript/baja/obj/dateTimeUtil', 'lex!baja,webEditors', 'jquery', 'moment', 'underscore', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/js/rc/asyncUtils/asyncUtils', 'css!nmodule/js/rc/pikaday/pikaday'], function (baja, types, dateTimeUtil, lexs, $, moment, _, BaseEditor, asyncUtils) {
  'use strict';

  var bajaLex = lexs[0],
      webEditorsLex = lexs[1],
      getDateOnlyFormat = dateTimeUtil.getDateOnlyFormat,
      DEFAULT_TIME_FORMAT = dateTimeUtil.DEFAULT_TIME_FORMAT,
      ISO_DATE_FORMAT = 'YYYY-MM-DD',
      // Ensure moment gets loaded/required before Pikaday
  doRequire = asyncUtils.doRequire,
      requirePikaday = _.once(function () {
    return doRequire('moment')["catch"](function (ignore) {}).then(function () {
      return doRequire('nmodule/js/rc/pikaday/pikaday');
    });
  });
  /**
   * Create a Pikaday i18n config from the baja and webEditors lexicons.
   *
   * @inner
   * @returns {Object}
   */


  var makeI18n = _.once(function () {
    function fromBajaLex(key) {
      return bajaLex.getSafe(key);
    }

    function toShort(key) {
      return key + '.short';
    }

    function getAllTags(range) {
      return _.map(range.getOrdinals(), function (o) {
        return range.getTag(o);
      });
    }

    var months = getAllTags(baja.$('baja:Month').getRange()),
        weekdays = getAllTags(baja.$('baja:Weekday').getRange());
    return {
      previousMonth: webEditorsLex.getSafe('DateEditor.Pikaday.previousMonth'),
      nextMonth: webEditorsLex.getSafe('DateEditor.Pikaday.nextMonth'),
      months: _.map(months, fromBajaLex),
      weekdays: _.map(weekdays, fromBajaLex),
      weekdaysShort: _.map(_.map(weekdays, toShort), fromBajaLex),
      firstDay: _.indexOf(weekdays, bajaLex.getSafe('weekday.firstDayOfWeek'))
    };
  });

  function toBajaDateArray(prop) {
    if (typeof prop === 'string') {
      return prop.split(';').map(function (str) {
        var m = moment(str, ISO_DATE_FORMAT);

        if (!m.isValid()) {
          throw new Error('invalid date format "' + str + '": ' + ISO_DATE_FORMAT + ' required');
        }

        return baja.Date.make({
          jsDate: m.toDate()
        });
      });
    }

    return prop || [];
  }

  function isEvent(el) {
    return el && el.classList.contains('has-event');
  }

  function applyEventRunClasses(row) {
    var kids = row.childNodes;

    _.each(kids, function (el, i) {
      if (isEvent(el)) {
        var wasEvent = isEvent(kids[i - 1]),
            willBeEvent = isEvent(kids[i + 1]);

        if (wasEvent && willBeEvent) {
          el.classList.add('event-run-middle');
        } else {
          if (!wasEvent) {
            el.classList.add('event-run-start');
          }

          if (!willBeEvent) {
            el.classList.add('event-run-end');
          }
        }
      }
    });
  }
  /**
   * `baja.Date` editor that utilizes the Pikaday date picker library.
   *
   * It supports the following `bajaux Properties`:
   *
   * - `disableDates`: dates to disable. Can be an array of `baja.Date`s or a
   *   semicolon-delimited string of dates in `YYYY-MM-DD` format.
   * - `embedded`: set to `true` to embed the date picker inline instead of
   *   popping up when the input field receives focus.
   * - `events`: dates to highlight as special/activated. Can be an array of
   *   `baja.Date`s or a semicolon-delimited string of dates in `YYYY-MM-DD`
   *   format.
   * - `onSelect`: callback that is called when selection changes.  If this callback
   *   is specified, PikadayDateEditor will not mark itself modified when the selection
   *   changes.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/fe/baja/compat/PikadayDateEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   *
   * @param {Object} opts
   * @param {Object} opts.properties
   * @param {Array.<baja.Date>} opts.properties.disableDates list of dates to disable
   * @param {Array.<baja.Date>} opts.properties.events list of dates that are events
   *  to be stylized differently.
   * @param {PikadayDateEditor~onSelectCallback} opts.properties.onSelect function that is called
   *  when the selection changes.
   *
   */


  var PikadayDateEditor = function PikadayDateEditor(opts) {
    BaseEditor.apply(this, arguments);
  };

  PikadayDateEditor.prototype = Object.create(BaseEditor.prototype);
  PikadayDateEditor.prototype.constructor = PikadayDateEditor;
  /**
   * @callback PikadayDateEditor~onSelectCallback
   * @param {baja.Date} date
   */

  /**
   * Initializes a text input to show the date and builds the Pikaday instance
   * to use. Arms an event handler to ensure that the editor is set modified
   * when a new date is selected.
   *
   * @param {JQuery} dom
   */

  PikadayDateEditor.prototype.doInitialize = function (dom) {
    var that = this,
        format = that.$getDateFormat(),
        disableDates = that.$getDisableDates(),
        embedded = that.$isEmbedded(),
        input = $('<input title="' + format + '" >').attr('type', embedded ? 'hidden' : 'text'),
        i18n = makeI18n();
    dom.append(input);
    return requirePikaday().then(function (Pikaday) {
      var pikadayOpts = {
        i18n: i18n,
        firstDay: i18n.firstDay,
        field: embedded ? undefined : input[0],
        events: that.$getEvents(),
        format: format,
        keyboardInput: !embedded,
        onDraw: function onDraw(picker) {
          var el = $(picker.el),
              editable = !that.isReadonly() && that.isEnabled(),
              enableHeader = editable && !that.$isSingleMonth(),
              tbody = el.find('tbody');

          while (tbody.length && tbody.children().length < 6) {
            //add blank rows so they vertically size the same
            tbody.append('<tr class="pika-row"><td class="is-empty"><button class="pika-button pika-day" type="button" disabled>&nbsp;</button></td></tr>');
          }

          _.each(el.find('.pika-row'), applyEventRunClasses);

          el.find('.pika-select-month').prop('disabled', !enableHeader);
          el.find('.pika-select-year').prop('disabled', !enableHeader || that.$isSingleMonth());
          el.find('.pika-prev').css('display', enableHeader ? '' : 'none');
          el.find('.pika-next').css('display', enableHeader ? '' : 'none');
        },
        onSelect: function onSelect() {
          var picker = that.$picker,
              selectionHandler = that.$getSelectionHandler();

          if (embedded) {
            that.$getInput().val(picker.toString());
          }

          if (selectionHandler) {
            var selectedDate = baja.Date.make({
              jsDate: picker.getDate()
            });
            selectionHandler.apply(that, [that, selectedDate]);
          } else {
            that.setModified(true);
          }
        },
        disableDayFn: function disableDayFn(checkDate) {
          if (!that.isEnabled()) {
            return true;
          }

          checkDate = moment(checkDate).format(ISO_DATE_FORMAT);
          return _.contains(disableDates, checkDate);
        }
      };
      var picker = that.$picker = new Pikaday(pikadayOpts);
      that.$origSetDate = picker.setDate;

      if (embedded) {
        dom.append(picker.el);
      }
    });
  };
  /**
   * Loads the date into the date picker.
   *
   * @param {baja.Date} date
   */


  PikadayDateEditor.prototype.doLoad = function (date) {
    var picker = this.$getPicker(),
        setDate = picker.setDate,
        jsDate = date.getJsDate(); // ensure that setting the date works even if readonly

    picker.setDate = this.$origSetDate;

    try {
      //picker.gotoDate(jsDate, true);
      picker.setDate(jsDate, true);
    } catch (e) {
      /*
      NCCB-19925: throws, but still sets date, on a disabled input field in FF
       */
      if (!date.equivalent(this.doRead())) {
        throw e;
      }
    } finally {
      picker.setDate = setDate;
    }
  };
  /**
   * Reads the currently selected date.
   *
   * @returns {baja.Date}
   */


  PikadayDateEditor.prototype.doRead = function () {
    return baja.Date.make({
      jsDate: this.$getPicker().getDate()
    });
  };
  /**
   * Destroys the backing Pikaday instance.
   */


  PikadayDateEditor.prototype.doDestroy = function () {
    return this.$getPicker().destroy();
  };
  /**
   * Sets picker and text input to readonly/editable.
   *
   * @param {Boolean} readonly
   */


  PikadayDateEditor.prototype.doReadonly = function (readonly) {
    this.$setEditable(!readonly && this.isEnabled());
    this.$getPicker().setDate = readonly ? _.noop : this.$origSetDate;
  };
  /**
   * Sets picker and text input to enabled/disabled.
   *
   * @param {Boolean} enabled
   */


  PikadayDateEditor.prototype.doEnabled = function (enabled) {
    this.$setEditable(enabled && !this.isReadonly());
  };
  /**
   * Clears the selection for the picker.
   */


  PikadayDateEditor.prototype.clearSelection = function () {
    this.$getPicker().setDate(null);
  };
  /**
   * @private
   * @returns {String} date format for printing dates
   */


  PikadayDateEditor.prototype.$getDateFormat = function () {
    var fmt = this.properties().getValue('timeFormat') || DEFAULT_TIME_FORMAT;
    return getDateOnlyFormat(fmt);
  };
  /**
   * @private
   * @returns {Function|null} onSelect handler
   */


  PikadayDateEditor.prototype.$getSelectionHandler = function () {
    return this.properties().getValue('onSelect');
  };
  /**
   * @private
   * @returns {boolean} indicator for a single month calendar
   */


  PikadayDateEditor.prototype.$isSingleMonth = function () {
    return this.properties().getValue('singleMonth');
  };
  /**
   * @private
   * @returns {Array.<String>} array of dates to disable, in YYYY-MM-DD format
   */


  PikadayDateEditor.prototype.$getDisableDates = function () {
    return toBajaDateArray(this.properties().getValue('disableDates')).map(function (date) {
      return moment(date.getJsDate()).format(ISO_DATE_FORMAT);
    });
  };
  /**
   * @private
   * @returns {Array.<String>} array of dates to highlight, in `#toDateString()`
   * format as per Pikaday reqs
   */


  PikadayDateEditor.prototype.$getEvents = function () {
    return toBajaDateArray(this.properties().getValue('events')).map(function (date) {
      return date.getJsDate().toDateString();
    });
  };
  /**
   * @private
   * @returns {jQuery} the text input that shows the date
   */


  PikadayDateEditor.prototype.$getInput = function () {
    return this.jq().children('input');
  };
  /**
   * @private
   * @returns {Pikaday} the Pikaday instance backing this editor
   */


  PikadayDateEditor.prototype.$getPicker = function () {
    return this.$picker;
  };
  /**
   * @private
   * @returns {boolean} true if the date picker should be embedded in place
   * instead of popping up on focus
   */


  PikadayDateEditor.prototype.$isEmbedded = function () {
    return !!this.properties().getValue('embedded');
  };
  /**
   * Update the date picker as to whether it should be editable or not.
   * @private
   * @param {boolean} editable
   */


  PikadayDateEditor.prototype.$setEditable = function (editable) {
    var picker = this.$getPicker();

    if (!this.$isEmbedded() && !editable) {
      picker.hide();
    } else {
      picker.draw();
    } //Pikaday listens for focus events. use disabled since it disallows focus


    this.$getInput().prop('disabled', !editable);
  };

  return PikadayDateEditor;
});
