/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/TimeEditor
 */
define(['baja!', 'jquery', 'Promise', 'bajaux/Widget', 'nmodule/webEditors/rc/fe/baja/BaseEditor'], function (baja, $, Promise, Widget, BaseEditor) {
  'use strict';

  var TimeFormat = baja.TimeFormat,
      SHOW_SECONDS = TimeFormat.SHOW_SECONDS,
      SHOW_MILLIS = TimeFormat.SHOW_MILLIS,
      NO_SECONDS_REGEX = /^\d\d:\d\d$/,
      NO_MILLIS_REGEX = /^\d\d:\d\d:\d\d$/;
  /**
   * A field editor for working with Times.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/TimeEditor
   */

  var TimeEditor = function TimeEditor() {
    BaseEditor.apply(this, arguments);
  };

  TimeEditor.prototype = Object.create(BaseEditor.prototype);
  TimeEditor.prototype.constructor = TimeEditor;
  /**
   * Returns the time input element.
   *
   * @private
   * @returns {JQuery}
   */

  TimeEditor.prototype.$getTimeElement = function () {
    return this.jq().children('input');
  };
  /**
   * Creates a time input element. (In browsers that do not support `time`
   * type input fields, this will result in a `text` input.)
   *
   * Depending on whether the `showSeconds` or `showMilliseconds` facets are
   * set on this editor, the `step` property of the input will be set to
   * show/hide the second and millisecond fields.
   *
   * @param {JQuery} dom
   */


  TimeEditor.prototype.doInitialize = function (dom) {
    var that = this,
        step = 1,
        props = that.properties(),
        showSeconds = props.getValue('showSeconds'),
        showMilliseconds = props.getValue('showMilliseconds');

    if (showSeconds === false) {
      step = 60;
    }

    if (showMilliseconds === true) {
      step = 0.001;
    }

    dom.append('<input type="time" step="' + step + '">');
    dom.on('change', 'input', function () {
      that.setModified(true);
    });
  };
  /**
   * Loads the time into the time input element. The value loaded may take the
   * form `HH:mm` if the `showSeconds` facet is false, `HH:mm:ss` (default), or
   * `HH:mm:ss.mmm` if `showMilliseconds` is true.
   *
   * @param {baja.Time} value
   * @returns {Promise} promise to be resolved when the value has loaded,
   * or rejected if the `baja.Time` could not be encoded to string
   */


  TimeEditor.prototype.doLoad = function (value) {
    var that = this,
        props = that.properties(),
        showSeconds = props.getValue('showSeconds'),
        showMilliseconds = props.getValue('showMilliseconds'),
        textPattern = 'HH:mm:ss',
        show = SHOW_SECONDS;

    if (showSeconds === false) {
      show = 0;
    }

    if (showMilliseconds) {
      show = SHOW_SECONDS | SHOW_MILLIS;
    }

    return value.toTimeString({
      textPattern: textPattern,
      show: show
    }).then(function (str) {
      that.$getTimeElement().val(str);
    });
  };
  /**
   * Reads the time value from the input element and creates a `baja.Time` from
   * it.
   *
   * @returns {Promise} promise to be resolved with the `baja.Time` read, or
   * rejected if the input contains an invalid time value
   */


  TimeEditor.prototype.doRead = function () {
    var val = this.$getTimeElement().val(); //baja.Time requires seconds and milliseconds fields.

    if (val.match(NO_SECONDS_REGEX)) {
      val += ':00';
    }

    if (val.match(NO_MILLIS_REGEX)) {
      val += '.000';
    }

    try {
      return baja.Time.DEFAULT.decodeFromString(val);
    } catch (e) {
      throw new Error('invalid Time value: "' + val + '"');
    }
  };
  /**
   * Enables or disables the time input element.
   *
   * @param {Boolean} enabled
   */


  TimeEditor.prototype.doEnabled = function (enabled) {
    this.$getTimeElement().prop('disabled', !enabled);
  };

  return TimeEditor;
});
