function _typeof(obj) { "@babel/helpers - typeof"; if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") { _typeof = function _typeof(obj) { return typeof obj; }; } else { _typeof = function _typeof(obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; }; } return _typeof(obj); }

/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/StringEditor
 */
define(['baja!', 'jquery', 'underscore', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/util/htmlUtils', 'hbs!nmodule/webEditors/rc/fe/baja/template/StringEditor'], function (baja, $, _, BaseEditor, htmlUtils, tplStringEditor) {
  'use strict';

  var DEFAULT_FIELD_WIDTH = 40,
      selectOnFocus = htmlUtils.selectOnFocus,
      selectAll = htmlUtils.selectAll;

  function valueToString(val) {
    if (baja.hasType(val, 'baja:Simple')) {
      return val.encodeToString();
    }

    return String(val);
  }

  function getNumber(num) {
    return baja.hasType(num, 'baja:Number') && num.valueOf();
  }

  function inRange(ed, val) {
    var length = val.length,
        props = ed.properties(),
        max = getNumber(props.getValue('max')),
        min = getNumber(props.getValue('min'));

    if (typeof max === 'number' && length > max) {
      throw new Error('length ' + length + ' > ' + max);
    } else if (typeof min === 'number' && length < min) {
      throw new Error('length ' + length + ' < ' + min);
    }
  }

  function validSlotName(ed, val) {
    var mustBeSlotName = ed.properties().getValue('mustBeSlotName');

    if (mustBeSlotName && !baja.SlotPath.isValidName(val)) {
      throw new Error('Illegal name "' + val + '"');
    }
  }
  /**
   * A field editor for working with strings.
   *
   * It accepts the following Properties:
   *
   * - `datalist`: (string|object) a semicolon-separated list of strings to use
   *   to generate an HTML `datalist` element. Alternatively, it can be an
   *   object literal from values to display strings; the display strings will
   *   be used to create the `datalist`, but `doRead()` will resolve the actual
   *   value if it corresponds to the entered display string. Useful for enum
   *   values with display tags, etc.
   * - `inputType`: (string) when not in multiline mode, set the `input` tag's
   *   `type` attribute. Defaults to `text`.
   * - `max`: (number) max length for the input.
   * - `min`: (number) min length for the input.
   * - `multiLine`: (boolean) if true, editor will be a `textarea`.
   * - `mustBeSlotName`: (boolean) input value must be a valid slot name.
   * - `placeholder`: (string) if given, the `placeholder` attribute of the
   *   text input will be set.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/StringEditor
   *
   * @param {baja.Facets} [params.facets] if a semicolon-separated string
   * `datalist` facet is provided, a `<datalist>` element will be appended for
   * autocomplete purposes.
   */


  var StringEditor = function StringEditor(params) {
    var that = this;
    BaseEditor.call(that, $.extend(true, {
      keyName: 'StringEditor'
    }, {
      properties: {
        inputType: {
          value: 'text',
          hidden: true,
          readonly: true
        }
      }
    }, params));
    /**
     * A StringEditor will fail to validate under the following conditions:
     *
     * - `max` facet is set and the current string value is too long
     * - `min` facet is set and the current string value is too short
     * - `mustBeSlotName` facet is set and the current string value is not a
     *   valid slot name
     * @method module:nmodule/webEditors/rc/fe/baja/StringEditor#validate
     */

    that.validators().add(function (val) {
      inRange(this, val);
    });
    that.validators().add(function (val) {
      validSlotName(this, val);
    });
    that.$prevText = '';
  };

  StringEditor.prototype = Object.create(BaseEditor.prototype);
  StringEditor.prototype.constructor = StringEditor;

  StringEditor.prototype.$isMultiLine = function () {
    return this.properties().getValue('multiLine');
  };

  StringEditor.prototype.$getTextElement = function () {
    var tag = this.$isMultiLine() ? 'textarea' : 'input';
    return this.jq().children(tag);
  };

  StringEditor.prototype.$getDatalistElement = function () {
    return this.jq().children('datalist');
  };

  StringEditor.prototype.$getDatalist = function () {
    return this.properties().getValue('datalist');
  };

  StringEditor.prototype.$selectAll = function () {
    selectAll(this.$getTextElement());
  };

  StringEditor.prototype.requestFocus = function () {
    BaseEditor.prototype.requestFocus.apply(this, arguments);
    this.$selectAll();
  };
  /**
   * Creates a text element. If the `multiLine` facet is set, will create a
   * `<textarea>` element (setting the `cols` property to the
   * `fieldWidth` facet, if present). Otherwise, creates
   * `<input type="text">`.
   *
   * @param {JQuery} dom
   */


  StringEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.on('input', 'input, textarea', function () {
      var val = this.value;

      if (val !== that.$prevText) {
        that.setModified(true);
      }

      that.$prevText = val;
    }).on('keyup', 'textarea', false);
    selectOnFocus(that, 'input', function () {
      return that.$selectAll();
    });
    return that.$rebuildHtml();
  };
  /**
   * Wipe and rebuild the contents of this editor's DOM. Why implement this
   * separately instead of directly in `doInitialize`? When the `datalist`
   * property changes, the safest/easiest thing to do is just to wipe and
   * rebuild with the correct `datalist` tag. This also gives subclasses
   * (namely, `RelevantStringEditor` a chance to get their own hooks in.
   *
   * @protected
   */


  StringEditor.prototype.$rebuildHtml = function () {
    var that = this,
        props = that.properties(),
        autocomplete = props.getValue('autocomplete'),
        datalist = that.$getDatalist(),
        fieldWidth = props.getValue('fieldWidth'),
        inputType = props.getValue('inputType') || 'text',
        placeholder = props.getValue('placeholder'),
        multiLine = that.$isMultiLine(),
        listId;

    if (autocomplete === 'off' && inputType === 'password') {
      autocomplete = 'new-password';
    } else if (autocomplete !== 'off') {
      autocomplete = '';
    }

    if (typeof fieldWidth !== 'number') {
      fieldWidth = DEFAULT_FIELD_WIDTH;
    }

    if (datalist) {
      if (typeof datalist === 'string') {
        datalist = datalist.split(';');
      } else {
        datalist = _.map(datalist, _.constant);
      }

      listId = that.generateId();
    }

    that.jq().html(tplStringEditor({
      multiLine: multiLine,
      fieldWidth: fieldWidth,
      inputType: inputType,
      autocomplete: autocomplete,
      placeholder: placeholder,
      datalist: datalist,
      listId: listId
    }));
  };
  /**
   * Loads the given string into the text element. If value is not already
   * a string, it will be toString-ed (or have `encodeToString` called if a
   * `baja:Simple`.
   *
   * @param {String} val
   */


  StringEditor.prototype.doLoad = function (val) {
    var str = valueToString(val),
        datalist = this.$getDatalist();

    if (datalist && _typeof(datalist) === 'object') {
      str = datalist[str] || str;
    }

    this.$prevText = str;
    this.$getTextElement().val(str);
  };
  /**
   * Reads the currently entered string from the text element. (If a value of
   * type `baja:Simple` was loaded, an instance of the same Type will be
   * decoded from the entered string and returned.)
   *
   * @returns {String} the currently entered string
   */


  StringEditor.prototype.doRead = function () {
    var value = this.value(),
        str = this.$getTextElement().val(),
        datalist = this.$getDatalist();

    if (datalist && _typeof(datalist) === 'object') {
      str = _.invert(datalist)[str] || str;
    }

    if (baja.hasType(value, 'baja:Simple')) {
      return baja.$(value.getType()).decodeFromString(str);
    }

    return str;
  };

  StringEditor.prototype.doChanged = function (prop) {
    if (prop === 'datalist') {
      return this.$rebuildHtml();
    }
  };
  /**
   * Enables or disables the text element.
   *
   * @param {Boolean} enabled
   */


  StringEditor.prototype.doEnabled = function (enabled) {
    this.$getTextElement().prop('disabled', !enabled);
  };
  /**
   * Sets the text element to readonly / not readonly.
   *
   * @param {Boolean} readonly
   */


  StringEditor.prototype.doReadonly = function (readonly) {
    this.$getTextElement().prop('readonly', readonly);
  };

  return StringEditor;
});
