/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/StatusValueEditor
 */
define(['baja!', 'log!nmodule.webEditors.rc.fe.baja.StatusValueEditor', 'jquery', 'Promise', 'underscore', 'bajaux/Properties', 'bajaux/events', 'bajaux/mixin/batchSaveMixin', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor', 'nmodule/webEditors/rc/fe/baja/DisplayOnlyEditor', 'nmodule/webEditors/rc/fe/baja/util/statusUtils'], function (baja, log, $, Promise, _, Properties, events, batchSaveMixin, BaseEditor, ComplexCompositeEditor, DisplayOnlyEditor, statusUtils) {
  'use strict';

  var MODIFY_EVENT = events.MODIFY_EVENT,
      LOAD_EVENT = events.LOAD_EVENT,
      COMMIT_READY = batchSaveMixin.COMMIT_READY,
      statusValueToString = statusUtils.statusValueToString,
      logSevere = log.severe.bind(log);
  /**
   * A composite editor for editing the writable's status and value.
   *
   * @private
   * @inner
   * @class
   */

  var StatusAndValue = function StatusAndValue(params) {
    ComplexCompositeEditor.apply(this, arguments);
  };

  StatusAndValue.prototype = Object.create(ComplexCompositeEditor.prototype);
  StatusAndValue.prototype.constructor = StatusAndValue;
  /**
   * Return the editor for the `status` slot.
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/StatusEditor}
   */

  StatusAndValue.prototype.$getStatusEditor = function () {
    return this.jq().children('.slot-status').data('widget');
  };
  /**
   * Return the editor for the `value` slot.
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/BaseEditor}
   */


  StatusAndValue.prototype.$getValueEditor = function () {
    return this.jq().children('.slot-value').data('widget');
  };
  /**
   * Use the `value` and `status` slots. Note that the `filter` facet will be
   * set on the `status` editor so that only the checkbox for `NULL` is shown.
   *
   * @returns {Array}
   */


  StatusAndValue.prototype.getSlotFilter = function () {
    return [{
      slot: 'value',
      properties: _.extend(this.properties().toObject(), {
        //if we define a uxFieldEditor to point to a StatusValueEditor
        //subclass, we don't want to propagate that down to the value editor.
        //infinite loop! other properties like min/max we keep.
        uxFieldEditor: null
      })
    }, {
      slot: 'status',
      properties: {
        filter: baja.Status.NULL
      }
    }];
  };
  /**
   * Arms an event handler that will disable the value editor when status
   * is set to null.
   *
   * @param {JQuery} dom
   */


  StatusAndValue.prototype.doInitialize = function (dom) {
    var that = this;
    dom.on(LOAD_EVENT + ' ' + MODIFY_EVENT, '.slot-status', function (e, ed) {
      that.doReadonly(that.isReadonly());
    });
    return ComplexCompositeEditor.prototype.doInitialize.apply(that, arguments);
  };
  /**
   * Since `CompositeEditor`s respect slot flags, and a `StatusValue` has its
   * `readonly` slot flag set by default, we manually re-enable the `status`
   * editor.
   *
   * @returns {Promise} call to super `load()` method, as normal
   */


  StatusAndValue.prototype.doLoad = function (statusValue) {
    var that = this;
    return ComplexCompositeEditor.prototype.doLoad.apply(that, arguments).then(function () {
      return that.doReadonly(that.isReadonly());
    });
  };
  /**
   * Enables/disables the status and value editors. Note that the value editor
   * will only be enabled if the status editor is not set to null.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */


  StatusAndValue.prototype.doReadonly = function (readonly) {
    var that = this,
        statusEd = that.$getStatusEditor(),
        valueEd = that.$getValueEditor();

    if (!statusEd) {
      return Promise.resolve();
    } //StatusValue has its status slot set to readonly by default. manually
    //re-enable it here.


    statusEd.setReadonly(readonly);
    return statusEd.read().then(function (status) {
      //enable value editor if status is not null.
      return valueEd.setReadonly(readonly || status.isNull());
    });
  };
  /**
   * A field editor for editing `StatusValue`s.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/StatusValueEditor
   */


  var StatusValueEditor = function StatusValueEditor(params) {
    BaseEditor.call(this, $.extend({
      keyName: 'StatusValueEditor'
    }, params));
    batchSaveMixin(this);
  };

  StatusValueEditor.prototype = Object.create(BaseEditor.prototype);
  StatusValueEditor.prototype.constructor = StatusValueEditor;
  /**
   * Return the element the editor for the actual status value (editable by the
   * user) lives in.
   * @private
   * @returns {jQuery}
   */

  StatusValueEditor.prototype.$getCompositeElement = function () {
    return this.jq().children('.composite');
  };
  /**
   * Return the editor for the actual status value (the one editable by the
   * user).
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/StatusEditor}
   */


  StatusValueEditor.prototype.$getCompositeEditor = function () {
    return this.$getCompositeElement().data('widget');
  };
  /**
   * Return the editor for the status display value (not editable
   * by the user).
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/BaseEditor}
   */


  StatusValueEditor.prototype.$getDisplayEditor = function () {
    return this.jq().children('.display').data('widget');
  };
  /**
   * Initialize the composite editor for editing the status and value, as well
   * as the display editor for showing the current status of the StatusValue.
   *
   * @param {JQuery} dom
   * @returns {Promise} promise to be resolved when the composite and
   * display editors have been initialized
   */


  StatusValueEditor.prototype.doInitialize = function (dom) {
    BaseEditor.prototype.doInitialize.apply(this, arguments);
    var that = this,
        compositeDom = $('<span class="composite"/>').appendTo(dom),
        displayDom = $('<span class="display"/>').appendTo(dom);
    dom.on(MODIFY_EVENT, '.composite', function () {
      that.$getCompositeEditor().read().then(function (diff) {
        var sn = that.value().newCopy();
        return diff.apply(sn).then(function () {
          return that.$updateDisplay(sn);
        });
      })["catch"](logSevere);
    });
    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });
    return Promise.all([new StatusAndValue({
      properties: that.properties().toObject()
    }).initialize(compositeDom), new DisplayOnlyEditor().initialize(displayDom)]);
  };
  /**
   * Loads the value into the composite editor for editing.
   *
   * @param {baja.Struct} statusValue `baja:StatusValue`
   * @returns {Promise} promise to be resolved when the composite editor
   * has finished loading
   */


  StatusValueEditor.prototype.doLoad = function (statusValue) {
    var that = this;
    return Promise.all([that.$getCompositeEditor().load(statusValue), that.$updateDisplay(statusValue)]);
  };
  /**
   * Updates the display editor showing the current status of the `StatusValue`.
   * Will be called whenever the editor has a new value loaded or is modified
   * by the user.
   *
   * @private
   * @param {baja.Struct} statusValue the `baja:StatusValue` being loaded or
   * modified
   * @returns {Promise} promise to be resolved when the display has
   * been updated
   */


  StatusValueEditor.prototype.$updateDisplay = function (statusValue) {
    if (!statusValue) {
      return Promise.reject(new Error('StatusValue required'));
    }

    var that = this;
    return statusValueToString(statusValue, that.properties().toValueMap()).then(function (str) {
      return that.$getDisplayEditor().load(str);
    });
  };

  StatusValueEditor.prototype.doEnabled = function (enabled) {
    return this.getChildWidgets().setAllEnabled(enabled);
  };

  StatusValueEditor.prototype.doReadonly = function (readonly) {
    return this.getChildWidgets().setAllReadonly(readonly);
  };

  StatusValueEditor.prototype.validate = function () {
    return this.$getCompositeEditor().validate();
  };

  StatusValueEditor.prototype.doRead = function () {
    return this.$getCompositeEditor().read();
  };

  StatusValueEditor.prototype.doSave = function (diff, params) {
    var progressCallback = params && params.progressCallback,
        prom = diff.apply(this.value(), params);

    if (progressCallback) {
      progressCallback(COMMIT_READY);
    }

    return prom;
  };

  StatusValueEditor.prototype.doDestroy = function () {
    return this.getChildWidgets().destroyAll();
  };

  return StatusValueEditor;
});
