/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/StatusEditor
 */
define(['baja!', 'lex!baja', 'log!nmodule.webEditors.rc.fe.baja.StatusEditor', 'jquery', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/util/statusUtils', 'hbs!nmodule/webEditors/rc/fe/baja/template/StatusEditor'], function (baja, lexs, log, $, BaseEditor, statusUtils, tplStatusEditor) {
  'use strict';

  var bajaLex = lexs[0],
      statusToString = statusUtils.statusToString,
      logSevere = log.severe.bind(log); //TODO: put this in BajaScript?

  var SUPPORT = {
    '1': 'disabled',
    '2': 'fault',
    '4': 'down',
    '8': 'alarm',
    '16': 'stale',
    '32': 'overridden',
    '64': 'null',
    '128': 'unackedAlarm'
  };
  /**
   * A field editor for working with Statuses. It supports the following
   * `Properties`:
   *
   * - `filter`: if a numeric `filter` property is provided, only those Statuses
   *   that bitwise-match the filter will be shown.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/StatusEditor
   *
   * @param {baja.Facets} [params.facets] if a numeric `filter` facet is
   * provided, only those Statuses that bitwise-match the filter will be
   * shown.
   */

  var StatusEditor = function StatusEditor(params) {
    BaseEditor.call(this, $.extend({
      keyName: 'StatusEditor'
    }, params));
  };

  StatusEditor.prototype = Object.create(BaseEditor.prototype);
  StatusEditor.prototype.constructor = StatusEditor;
  /**
   * Return the text box that shows the string representation of the status
   * when the editor is set to readonly.
   *
   * @private
   * @returns {jQuery}
   */

  StatusEditor.prototype.$getReadonlyTextInput = function () {
    return this.jq().find('input[type=text]');
  };
  /**
   * Get statuses matching the given filter (0 = no filter).
   *
   * @private
   * @param {Number} [filter]
   * @returns {Array} array of objects with `name`, `display`, `bit` properties
   * to be loaded into Handlebars template
   */


  StatusEditor.prototype.$getStatuses = function (filter) {
    var statuses = [],
        bit = 1,
        name;

    while (name = SUPPORT[bit]) {
      // isToDisplay will check if the bit to be checked/unchecked can be displayed by the field editor
      if (this.isToDisplay(bit) && (!filter || filter & bit)) {
        statuses.push({
          name: name,
          display: bajaLex.get('Status.' + name),
          bit: bit,
          id: this.generateId()
        });
      }

      bit <<= 1;
    }

    return statuses;
  };
  /**
   * Creates checkboxes and labels for all available Status options. If no
   * `filter` facet is given, this will be all eight statuses from `disabled`
   * to `unackedAlarm`.
   *
   * @param {JQuery} dom
   */


  StatusEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.html(tplStatusEditor({
      statuses: that.$getStatuses(that.properties().getValue('filter'))
    }));
    dom.on('change', 'input', function () {
      that.setModified(true);
      that.read().then(statusToString).then(function (str) {
        that.$getReadonlyTextInput().val(str);
      })["catch"](logSevere);
    });
  };
  /**
   * Checks/unchecks the appropriate checkboxes to match the bits set on the
   * input Status.
   *
   * @param {baja.Status} value
   */


  StatusEditor.prototype.doLoad = function (value) {
    var that = this,
        dom = that.jq(),
        bits = value.getBits(),
        bit = 1,
        inputs = dom.find('input'),
        name;
    inputs.prop('checked', false);

    while (name = SUPPORT[bit]) {
      // isToDisplay will check if the bit to be checked/unchecked can be displayed by the field editor.
      if (that.isToDisplay(bit) && bit & bits) {
        inputs.filter('[name=' + name + ']').prop('checked', true);
      }

      bit <<= 1;
    }

    return statusToString(value).then(function (str) {
      that.$getReadonlyTextInput().val(str);
    });
  };
  /**
   * Will return true by default, as the status editor will display all bits.
   * @param {integer} bit
   * @return {boolean} true is returned, there by states like disabled, fault, down, alarm, overriden, unackedAlarm, null and stale will be supported.
   */


  StatusEditor.prototype.isToDisplay = function (bit) {
    return true;
  };
  /**
   * Reads which checkboxes are currently checked, and assembles a `baja.Status`
   * instance with bits set accordingly.
   *
   * @returns {baja.Status}
   */


  StatusEditor.prototype.doRead = function () {
    var dom = this.jq(),
        bits = 0;
    dom.find('input[type=checkbox]').each(function (i, input) {
      var $input = $(input);

      if ($input.prop('checked')) {
        bits |= parseInt($input.data('bit'), 10);
      }
    });
    return baja.Status.make(bits);
  };

  function setEditable(ed, editable) {
    ed.jq().find('input[type=checkbox]').prop('disabled', !editable);
  }
  /**
   * Enables or disables all checkboxes.
   *
   * @param {Boolean} enabled
   */


  StatusEditor.prototype.doEnabled = function (enabled) {
    setEditable(this, !this.isReadonly() && enabled);
  };
  /**
   * Disables or enables all checkboxes.
   *
   * @param {Boolean} readonly
   */


  StatusEditor.prototype.doReadonly = function (readonly) {
    setEditable(this, this.isEnabled() && !readonly);
  };

  return StatusEditor;
});
