/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/PopOutEditor
 */
define(['lex!webEditors', 'log!nmodule.webEditors.rc.fe.baja.PopOutEditor', 'bajaux/Widget', 'bajaux/events', 'jquery', 'nmodule/webEditors/rc/fe/BaseWidget', 'nmodule/webEditors/rc/wb/commands/PopOutCommand'], function (lexs, log, Widget, events, $, BaseWidget, PopOutCommand) {
  'use strict';

  var webEditorsLex = lexs[0],
      logError = log.severe.bind(log),
      COMMAND_CHANGE_EVENT = events.command.CHANGE_EVENT;
  /**
   * This editor functions as a "watcher" on another field editor (typically of
   * "mini" form factor), for the purpose of popping out a compact editor for
   * more detailed editing.
   *
   * The work of this editor is delegated out to `PopOutCommand` - consider
   * using that directly.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/BaseWidget
   * @alias module:nmodule/webEditors/rc/fe/baja/PopOutEditor
   *
   * @see module:nmodule/webEditors/rc/wb/commands/PopOutCommand
   */

  var PopOutEditor = function PopOutEditor() {
    var that = this,
        cmd = new PopOutCommand();
    BaseWidget.apply(that); //no params?

    cmd.on(COMMAND_CHANGE_EVENT, function () {
      //TODO: remove/improve this hack.
      //in a unit test, the Command gets created, the editor gets created
      //and destroyed, then the promise to resolve the Command's display name
      //is resolved afterwards, triggering the change and the annoying
      //"already destroyed" spam.
      if (!that.isDestroyed()) {
        that.setEnabled(cmd.isEnabled())["catch"](logError);
      }
    });
    that.getCommandGroup().add(cmd);
  };

  PopOutEditor.prototype = Object.create(BaseWidget.prototype);
  PopOutEditor.prototype.constructor = PopOutEditor;
  /**
   * Returns the `PopOutCommand`.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/commands/PopOutCommand}
   */

  PopOutEditor.prototype.$getCommand = function () {
    // noinspection JSValidateTypes
    return this.getCommandGroup().get(0);
  };
  /**
   * Returns the button element.
   *
   * @private
   * @returns {jQuery}
   */


  PopOutEditor.prototype.$getButton = function () {
    return this.jq().children('button');
  };
  /**
   * Create the button element to display. Override this to change the
   * appearance of the editor as needed, but ensure that the returned element
   * is of `button` type so click event handlers function properly.
   *
   * @returns {jQuery} a jQuery element of type `button`
   */


  PopOutEditor.prototype.makeButton = function () {
    return $('<button type="button" class="ux-btn-tb" disabled="disabled"/>').text(webEditorsLex.get('popOut'));
  };
  /**
   * Appends a pop-out button to the element and arms appropriate event
   * handlers, including a click handler on the button that will show the
   * compact editor dialog.
   *
   * @param {JQuery} dom
   */


  PopOutEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.append(that.makeButton());
    dom.on('click', 'button', function () {
      that.$getCommand().invoke()["catch"](logError);
    });
    that.setEnabled(that.$getCommand().isEnabled())["catch"](logError);
    return null; //squelch "promise not returned"
  };
  /**
   * Load in an editor to watch.
   * @param {module:bajaux/Widget|null} ed
   */


  PopOutEditor.prototype.doLoad = function (ed) {
    if (ed instanceof Widget) {
      this.$getCommand().setWatchedEditor(ed);
    }
  };

  PopOutEditor.prototype.doRead = function () {
    var ed = this.value();
    return ed ? ed.read() : null;
  };
  /**
   * Enables/disables the pop-out button.
   *
   * @param {Boolean} enabled
   */


  PopOutEditor.prototype.doEnabled = function (enabled) {
    var clickable = !this.isReadonly() && enabled;
    this.$getButton().prop('disabled', !clickable).toggle(clickable);
  };
  /**
   * Disables/enables the pop-out button.
   *
   * @param {Boolean} readonly
   */


  PopOutEditor.prototype.doReadonly = function (readonly) {
    var clickable = this.isEnabled() && !readonly;
    this.$getButton().prop('disabled', !clickable).toggle(clickable);
  };

  return PopOutEditor;
});
