/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/OrdListEditor
 */
define(['baja!', 'lex!webEditors', 'bajaux/Widget', 'bajaux/commands/Command', 'bajaux/util/CommandButtonGroup', 'bajaScript/baja/ord/OrdList', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/fe/array/ArrayCompositeEditor', 'nmodule/webEditors/rc/fe/baja/DisplayOnlyEditor', 'nmodule/webEditors/rc/fe/config/CompositeBuilder', 'nmodule/webEditors/rc/wb/links/OrdChooser'], function (baja, lexs, Widget, Command, CommandButtonGroup, OrdList, CompositeEditor, feDialogs, ArrayCompositeEditor, DisplayOnlyEditor, CompositeBuilder, OrdChooser) {
  'use strict';

  var webEditorsLex = lexs[0]; ////////////////////////////////////////////////////////////////
  // Custom Commands
  ////////////////////////////////////////////////////////////////

  /**
   * Command to add a new ORD.
   *
   * @inner
   * @class
   * @extends module:bajaux/commands/Command
   * @param {module:nmodule/webEditors/rc/fe/baja/OrdListEditor} ed
   */

  var AddCommand = function AddCommand(ed) {
    Command.call(this, {
      module: 'webEditors',
      lex: 'commands.addItem',
      func: function func() {
        return feDialogs.showFor({
          type: OrdChooser,
          formFactor: 'compact',
          value: baja.Ord.make()
        }).then(function (ord) {
          if (!ord) {
            return;
          }

          return ed.read().then(function (ol) {
            return ed.$getArrayEditor().load(ol.getOrds().concat([ord]));
          });
        });
      }
    });
  };

  AddCommand.prototype = Object.create(Command.prototype);
  AddCommand.prototype.constructor = AddCommand; ////////////////////////////////////////////////////////////////
  // OrdListEditor
  ////////////////////////////////////////////////////////////////

  /**
   * Editor for handling `baja:OrdList` values.
   *
   * When form factor is `mini`, this will be a display-only editor. If the
   * multiLine property is true then it will display each ORD on its own line.
   * If the multiLine property is false then if there is only one ORD it will
   * display that ORD. Otherwise, it will display the number of ORDs in the
   * list.
   *
   * When form factor is `compact`, this will be an editor allowing you to edit
   * all the individual ORDs, add and remove them, and change their order.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/OrdListEditor
   */

  var OrdListEditor = function OrdListEditor(params) {
    var that = this;
    CompositeEditor.apply(that, arguments);

    if (that.getFormFactor() === Widget.formfactor.mini) {
      DisplayOnlyEditor.$mixin(this);
    }

    var getEd = function getEd() {
      return that.$getArrayEditor();
    };

    that.getCommandGroup().add(new AddCommand(that), new ArrayCompositeEditor.RemoveCommand(getEd), new ArrayCompositeEditor.MoveUpCommand(getEd), new ArrayCompositeEditor.MoveDownCommand(getEd));
  };

  OrdListEditor.prototype = Object.create(CompositeEditor.prototype);
  OrdListEditor.prototype.constructor = OrdListEditor;
  /**
   * Configures the composite builder for the loaded OrdList
   * instance.
   *
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */

  OrdListEditor.prototype.makeBuilder = function () {
    var that = this,
        builder = new CompositeBuilder();

    builder.getKeys = function () {
      return ['array', 'commands'];
    };

    builder.getValueFor = function (key) {
      switch (key) {
        case 'array':
          return this.getDataSource().getOrds();

        case 'commands':
          return that.getCommandGroup();
      }
    };

    builder.getConfigFor = function (key) {
      switch (key) {
        case 'array':
          return {
            type: ArrayCompositeEditor,
            properties: {
              numbers: {
                show: true,
                countFromZero: false
              }
            }
          };

        case 'commands':
          return {
            type: CommandButtonGroup
          };
      }
    };

    builder.getDomFor = function (key) {
      return that.jq().children('.js-' + key);
    };

    return builder;
  };
  /**
   * Create the DOM structure for housing the ArrayCompositeEditor and
   * the CommandButtonGroup.
   *
   * @param {JQuery} dom
   * @returns {Promise}
   */


  OrdListEditor.prototype.doInitialize = function (dom) {
    dom.html('<div class="js-array"/><div class="editor js-commands"/>');
    return CompositeEditor.prototype.doInitialize.apply(this, arguments);
  };
  /**
   * Read a new `baja:OrdList` instance from the currently entered ORDs.
   *
   * @returns {Promise.<baja.OrdList>}
   */


  OrdListEditor.prototype.doRead = function () {
    return this.$getArrayEditor().read().then(function (array) {
      return OrdList.make(array);
    });
  };
  /**
   * When in mini/display-only mode:
   * If multiLine property = true: display each ORD on its own line.
   * If multiLine property = false or non-existent: if there is only one ORD
   * then display it, otherwise display the number of ORDs in the list.
   *
   * @returns {String}
   */


  OrdListEditor.prototype.valueToString = function (ordList) {
    var size = ordList.size();
    var multiLine = !!this.properties().getValue('multiLine');
    this.jq().toggleClass('multiline', multiLine);

    if (multiLine) {
      return ordList.getOrds().map(String).join('\n');
    } else {
      if (size === 1) {
        return ordList.getOrds()[0].toString();
      } else {
        return webEditorsLex.get('OrdListEditor.ords', size);
      }
    }
  };
  /**
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/array/ArrayCompositeEditor}
   */


  OrdListEditor.prototype.$getArrayEditor = function () {
    // noinspection JSValidateTypes
    return this.getBuilder().getEditorFor('array');
  };
  /**
   * @private
   * @returns {module:bajaux/commands/Command}
   */


  OrdListEditor.prototype.$getAddCommand = function () {
    return this.getCommandGroup().get(0);
  };

  return OrdListEditor;
});
