/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/DaysOfWeekBitsEditor
 */
define(['baja!', 'baja!baja:Weekday', 'lex!baja', 'bajaux/events', 'jquery', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/baja/BaseEditor'], function (baja, types, lexs, events, $, Promise, _, fe, BaseEditor) {
  'use strict';

  var bajaLex = lexs[0],
      WEEKDAY_RANGE = baja.$('baja:Weekday').getRange(),
      MODIFY_EVENT = events.MODIFY_EVENT;
  /**
   * Editor for working with `baja:DaysOfWeekBits` values.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/DaysOfWeekBitsEditor
   */

  var DaysOfWeekBitsEditor = function DaysOfWeekBitsEditor(params) {
    BaseEditor.apply(this, arguments);
  };

  DaysOfWeekBitsEditor.prototype = Object.create(BaseEditor.prototype);
  DaysOfWeekBitsEditor.prototype.constructor = DaysOfWeekBitsEditor;
  /**
   * Initialize checkboxes for each day of the week.
   *
   * @param {jQuery} dom
   */

  DaysOfWeekBitsEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });
    return Promise.all(_.map(WEEKDAY_RANGE.getOrdinals(), function (i) {
      var display = bajaLex.get(WEEKDAY_RANGE.getTag(i) + '.short');
      return fe.buildFor({
        dom: $('<span data-bit="' + i + '"/>').appendTo(dom),
        properties: {
          trueText: display,
          falseText: display
        },
        value: false,
        formFactor: 'mini'
      });
    }));
  };
  /**
   * Set the values of the individual day-of-week checkboxes according to
   * whether the corresponding bits are set in the loaded value.
   *
   * @param {baja.Simple} bits a `baja:DaysOfWeekBits` value
   */


  DaysOfWeekBitsEditor.prototype.doLoad = function (bits) {
    var bitNum = parseInt(bits.encodeToString(), 16);
    return Promise.all(_.map(this.getChildEditors(), function (kid, i) {
      var isSet = bitNum & 1 << i;
      return kid.load(!!isSet);
    }));
  };
  /**
   * Resolve a `baja:DaysOfWeekBits` value with the bits set according to the
   * current values of the checkboxes.
   *
   * @returns {Promise}
   */


  DaysOfWeekBitsEditor.prototype.doRead = function () {
    return this.getChildEditors().readAll().then(function (booleans) {
      var bits = _.reduce(booleans, function (memo, checked, i) {
        return checked ? memo + (1 << i) : memo;
      }, 0);

      return baja.$('baja:DaysOfWeekBits', bits);
    });
  };
  /**
   * Destroy all child editors.
   *
   * @returns {Promise}
   */


  DaysOfWeekBitsEditor.prototype.doDestroy = function () {
    return this.getChildEditors().destroyAll();
  };
  /**
   * Set all child editors readonly/writable.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */


  DaysOfWeekBitsEditor.prototype.doReadonly = function (readonly) {
    return this.getChildEditors().setAllReadonly(readonly);
  };
  /**
   * Set all child editors enabled/disabled.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */


  DaysOfWeekBitsEditor.prototype.doEnabled = function (enabled) {
    return this.getChildEditors().setAllEnabled(enabled);
  };

  return DaysOfWeekBitsEditor;
});
