/*
 * Copyright 2019 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/CertificateEditor
 */
define(['baja!', 'lex!webEditors', 'Promise', 'nmodule/webEditors/rc/fe/baja/FileInputEditor', 'nmodule/webEditors/rc/util/Switchboard', 'hbs!nmodule/webEditors/rc/fe/baja/template/CertificateEditor', 'baja!baja:X509Certificate'], function (baja, lexicons, Promise, FileInputEditor, Switchboard, template) {
  'use strict';
  /*eslint-env browser */

  /*jshint browser: true */

  var webEditorsLex = lexicons[0];
  /**
   * An editor for working with `baja:X509Certificate` instances.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/FileInputEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/CertificateEditor
   */

  var CertificateEditor = function CertificateEditor() {
    var that = this;
    FileInputEditor.apply(this, arguments);
    that.validators().add(function (x509cert) {
      var certText = x509cert.encodeToString();

      if (certText !== '' && !CertificateEditor.$isCertificateValid(certText)) {
        throw new Error(webEditorsLex.get('CertificateEditor.invalidCert'));
      }
    });
    new Switchboard(this).allow('$getLastFileContents').oneAtATime();
  }; // Extend and set up prototype chain


  CertificateEditor.prototype = Object.create(FileInputEditor.prototype);
  CertificateEditor.prototype.constructor = CertificateEditor;
  /**
   * Initialize the field editor. Add a label to the file input.
   *
   * @param {JQuery} dom
   */

  CertificateEditor.prototype.doInitialize = function (dom) {
    FileInputEditor.prototype.doInitialize.call(this, dom);
    dom.prepend(template());
  };
  /**
   * Loads the SubjectDN of the certificate into the field editor.
   *
   * @param {baja.Simple} x509cert the `baja:X509Certificate` certificate to load
   */


  CertificateEditor.prototype.doLoad = function (x509cert) {
    this.jq().find('input[type=file]').val('');
    this.$lastChangedFile = undefined;
    this.$lastReadFile = undefined;
    this.$lastFileContents = undefined;
    return this.$updateSubjectDn(x509cert);
  };
  /**
   * @private
   * @param {baja.Simple} x509cert
   * @returns {Promise}
   */


  CertificateEditor.prototype.$updateSubjectDn = function (x509cert) {
    var subjectDnLabel = this.jq().find('.certificate-subject-dn-label');
    var warningLabel = this.jq().find('.certificate-warning-label');

    if (!x509cert || x509cert.valueOf() === '') {
      subjectDnLabel.text(webEditorsLex.get('CertificateEditor.subjectDn') + ': ' + webEditorsLex.get('CertificateEditor.noCertificate'));
      warningLabel.text('');
      return Promise.resolve();
    }

    var subjectDN = baja.rpc('type:baja:X509Certificate', 'getSubjectDn', x509cert.encodeToString());
    return subjectDN.then(function (result) {
      subjectDnLabel.text(webEditorsLex.get('CertificateEditor.subjectDn') + ': ' + result);
    })["catch"](function (error) {
      subjectDnLabel.text(error.message);
    });
  };

  CertificateEditor.prototype.doModified = function (modified) {
    var that = this;
    var warningDnLabel = that.jq().find('.certificate-warning-label');

    if (modified === true) {
      return that.$getLastFileContents().then(function (text) {
        return that.$updateSubjectDn(text);
      }).then(function () {
        return that.$getWarningText();
      }).then(function (warningText) {
        return warningDnLabel.text(warningText);
      });
    } else {
      warningDnLabel.text('');
    }
  }; //TODO: this moves to FileInputEditor

  /**
   * @private
   * @returns {Promise.<string>} the contents of the last selected file
   */


  CertificateEditor.prototype.$getLastFileContents = function () {
    FileInputEditor.prototype.doRead.apply(this, arguments);
    var that = this,
        lastUploadedFile = FileInputEditor.getLastUploadedFile(),
        lastReadFile = that.$lastReadFile;

    if (lastReadFile === lastUploadedFile) {
      return Promise.resolve(that.$lastFileContents);
    }

    return CertificateEditor.$readFileAsText(lastUploadedFile).then(function (text) {
      that.$lastFileContents = text;
      that.$lastReadFile = lastUploadedFile;
      return text;
    })["catch"](function () {
      throw new Error(webEditorsLex.get('CertificateEditor.failedRead'));
    });
  };
  /**
   * Override FileInputEditor's doRead to jibe with default Widget#doRead.
   */


  CertificateEditor.prototype.doRead = function () {
    return this.$getLastFileContents().then(function (text) {
      return baja.$('baja:X509Certificate', text);
    });
  }; //TODO: this moves to FileInputEditor

  /**
   * Read the file and process successful and unsuccessful reads
   *
   * @private
   * @param {File} certFile the file to upload
   * @returns {Promise}
   */


  CertificateEditor.$readFileAsText = function (certFile) {
    var reader = new FileReader(); // eslint-disable-next-line promise/avoid-new

    return new Promise(function (resolve, reject) {
      reader.onload = function () {
        resolve(reader.result);
      };

      reader.onerror = function () {
        reject(reader.error);
      };

      reader.readAsText(certFile);
    });
  };
  /**
   * Saves the value of the selected file back to the server. Does some
   * simple validation to make sure the file is a PEM certificate.
   *
   * @private
   * @param {string} certificate a string representing the certificate (PEM format)
   */


  CertificateEditor.$isCertificateValid = function (certificate) {
    var trimmedCert = certificate.trim();
    return trimmedCert.indexOf('-----BEGIN CERTIFICATE-----') === 0 && trimmedCert.indexOf('-----END CERTIFICATE-----') !== -1;
  };
  /**
   * Retrieve editor warning text from facet.
   *
   * @private
   * @returns {Promise} promise resolving to editor warning text
   */


  CertificateEditor.prototype.$getWarningText = function () {
    var warningTextPattern = this.properties().getValue('warningText') || '';
    return baja.Format.format({
      pattern: warningTextPattern
    });
  };

  return CertificateEditor;
});
