/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*jshint devel: true */
define(['baja!', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/typeUtils'], function (baja, _, typeUtils) {
  'use strict';

  var isComplex = typeUtils.isComplex,
      isComponent = typeUtils.isComponent;

  function validateMap(map) {
    if (!map) {
      throw new Error('mapping required');
    }

    _.each(map, function (target) {
      if (!(typeof target === 'string' || target instanceof baja.Property || baja.hasType(target, 'baja:Facets'))) {
        throw new Error('can only map slots or Facets instances');
      }
    });
  }

  function getSources(comp, targetSlotName, map) {
    return _.chain(map).map(function (targetSlot, regex) {
      if (String(targetSlot) === targetSlotName) {
        return comp.getSlots().filter(function (slot) {
          return String(slot).match(regex);
        }).toArray();
      }
    }).compact().flatten().value();
  }

  function getTarget(comp, srcSlotName, map) {
    var targetSlot = _.find(map, function (targetSlot, regex) {
      return srcSlotName.match(regex);
    });

    if (targetSlot) {
      if (baja.hasType(targetSlot, 'baja:Facets')) {
        return targetSlot;
      }

      return comp.getSlot(targetSlot);
    }
  }

  function getTargetFacets(comp, slot, map) {
    var target = getTarget(comp, String(slot), map);

    if (target) {
      if (target instanceof baja.Property && target.getType().is('baja:Facets')) {
        return comp.get(target);
      } else if (baja.hasType(target, 'baja:Facets')) {
        return target;
      }
    }
  }
  /**
   * API Status: **Private**
   *
   * Misc. BajaScript-related utility functions.
   *
   * @exports nmodule/webEditors/rc/baja/bajaUtils
   */


  var exports = {}; //TODO: either this goes into core BajaScript, or ControlPoint comes out. see kitControl:Logic.

  /**
   * Declaratively express overridden `getSlotFacets()` behavior on a
   * BajaScript Complex object. This will do two things:
   *
   * @param {Complex} comp a `Complex` instance
   * @param {Object} map mapping of slot name regexes to either target slots
   * (`String|baja.Slot`) or `baja.Facets` instances
   *
   * @example
   *   <caption>When getting slot facets for the "out" slot, return the value
   *   of the "facets" property instead.</caption>
   *   proxySlotFacets(myPoint, {
   *     '^out$': 'facets'
   *   });
   *
   * @example
   *   <caption>When getting slot facets for the "in" slot, return a hardcoded
   *   Facets instance.</caption>
   *   proxySlotFacets(myPoint, {
   *     '^in$': baja.Facets.make({ value: 'myFacets' });
   *   });
   */

  exports.proxySlotFacets = function (comp, map) {
    if (!isComplex(comp)) {
      throw new Error('baja.Complex instance required');
    }

    validateMap(map);
    var _getFacets = comp.getFacets;

    comp.getFacets = function (slot) {
      return getTargetFacets(this, slot, map) || _getFacets.apply(this, arguments);
    };

    if (isComponent(comp)) {
      comp.attach('changed', function (slot, cx) {
        _.each(getSources(comp, String(slot), map), function (sourceSlot) {
          comp.fireHandlers('facetsChanged', baja.error, comp, sourceSlot, cx);
        });
      });
    }
  };

  return exports;
});
