/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnetAws/rc/wb/mgr/BacnetAwsDeviceUxManager
 */
define(['baja!',
  'underscore',
  'nmodule/bacnet/rc/wb/mgr/model/BacnetDeviceMgrModel',
  'nmodule/webEditors/rc/fe/baja/util/compUtils',
  'nmodule/webEditors/rc/wb/mgr/mgrUtils',
  'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo',
  'nmodule/bacnetAws/rc/wb/mgr/commands/DeviceCommunicationControlCommand',
  'nmodule/bacnetAws/rc/wb/mgr/commands/ReinitializeDeviceCommand',
  'nmodule/bacnetAws/rc/wb/mgr/commands/BackupCommand',
  'nmodule/bacnetAws/rc/wb/mgr/commands/RestoreCommand',
  'nmodule/bacnetOws/rc/wb/mgr/BacnetOwsDeviceUxManager',
  'nmodule/webEditors/rc/servlets/password',
  'css!nmodule/bacnetAws/rc/bacnetAws',
  'baja!bacnet:BacnetAddress,' +
  'bacnet:BacnetDevice,' +
  'bacnet:BacnetDeviceFolder,' +
  'bacnet:BacnetDeviceObject,' +
  'bacnet:BacnetObjectIdentifier,' +
  'bacnet:BacnetObjectType,' +
  'bacnetAws:BacnetAwsDevice,' +
  'bacnetAws:BacnetAwsDeviceFolder,' +
  'bacnet:DiscoveryDevice'], function (baja,
                                       _,
                                       BacnetDeviceMgrModel,
                                       compUtils,
                                       mgrUtils,
                                       MgrTypeInfo,
                                       DeviceCommunicationControlCommand,
                                       ReinitializeDeviceCommand,
                                       BackupCommand,
                                       RestoreCommand,
                                       BacnetOwsDeviceUxManager,
                                       password) {

  'use strict';

  var BACNET_AWS_FOLDER_TYPE = baja.lt('bacnetAws:BacnetAwsDeviceFolder'),
      BACNET_AWS_DEVICE_TYPE = baja.lt('bacnetAws:BacnetAwsDevice'),
      getMainTableSelection  = mgrUtils.getMainTableSelectedSubjects;

////////////////////////////////////////////////////////////////
// BacnetAwsDeviceUxManager
////////////////////////////////////////////////////////////////

  /**
   * BacnetAwsDeviceUxManager constructor.
   *
   * @class
   * @alias module:nmodule/bacnetAws/rc/wb/mgr/BacnetAwsDeviceUxManager
   * @extends module:nmodule/bacnetOws/rc/wb/mgr/BacnetOwsDeviceUxManager
   * @param {Object} params
   */
  var BacnetAwsDeviceUxManager = function BacnetAwsDeviceUxManager(params) {
    BacnetOwsDeviceUxManager.call(this, _.extend({
      moduleName: 'bacnetAws',
      keyName: 'BacnetAwsDeviceUxManager',
      folderType: BACNET_AWS_FOLDER_TYPE
    }, params));
  };

  BacnetAwsDeviceUxManager.prototype = Object.create(BacnetOwsDeviceUxManager.prototype);
  BacnetAwsDeviceUxManager.prototype.constructor = BacnetAwsDeviceUxManager;

  /**
   * Function to return an array of types to be offered upon the execution
   * of the command to add new instances.
   *
   * @returns {Array.<baja.Type>}
   */
  BacnetAwsDeviceUxManager.prototype.getNewTypes = function () {
    return [BACNET_AWS_DEVICE_TYPE];
  };

  /**
   * Return the BacnetAwsNetwork from the source component (which may be the network or may
   * be a device folder several levels deep).
   *
   * @returns {baja.Component}
   */
  BacnetAwsDeviceUxManager.prototype.getNetwork = function () {
    var model = this.getModel(),
        comp  = model.getComponentSource().getContainer();

    return compUtils.closest(comp, baja.lt('bacnetAws:BacnetAwsNetwork'));
  };

  /**
   * Return the types available to be created from a given discovery item. In the
   * case of the BACnet device manager, the only option is a new BBacnetDevice.
   *
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>}
   */
  BacnetAwsDeviceUxManager.prototype.getTypesForDiscoverySubject = function () {
    return MgrTypeInfo.make(BACNET_AWS_DEVICE_TYPE);
  };

  /**
   * Makes the model for the main database table.
   *
   * @param {baja.Component} component - a bacnet network or bacnet device folder instance.
   * @returns {Promise.<module:nmodule/driver/rc/wb/mgr/DeviceMgrModel>}
   */
  BacnetAwsDeviceUxManager.prototype.makeModel = function (component) {
    return BacnetDeviceMgrModel.make(component,
      [ BACNET_AWS_DEVICE_TYPE ],
      BACNET_AWS_FOLDER_TYPE);
  };

  /**
   * Test that we have a single item selected in the main database table,
   * and that it is a device, not a folder.
   *
   * @param mgr - the device manager instance
   * @returns {Boolean}
   */
  function isSingleDeviceSelected(mgr) {
    var subjects = getMainTableSelection(mgr);
    return (subjects.length === 1) && (subjects[0].getType().is('bacnetAws:BacnetAwsDevice'));
  }

  /**
   * Load the network or device folder into the widget. This will set up
   * a handler to watch the main table selection, and enable or disable the
   * Commands based on the presence of at least one selected
   * device.
   *
   * @returns {Promise}
   */
  BacnetAwsDeviceUxManager.prototype.doLoad = function () {
    var that = this,
        args = arguments;
     return password.$isSecure()
       .then(function (secure) {
        return BacnetOwsDeviceUxManager.prototype.doLoad.apply(that, args)
          .then(function () {
             if (!secure) {
               return; //since all commands require a password, leave them disabled
             }
            var selection = mgrUtils.getMainTableSelection(that);
            selection.on('changed', function () {
              var command = mgrUtils.findCommand(that, DeviceCommunicationControlCommand);
              command.setEnabled(isSingleDeviceSelected(that));
              command = mgrUtils.findCommand(that, ReinitializeDeviceCommand);
              command.setEnabled(isSingleDeviceSelected(that));
              command = mgrUtils.findCommand(that, BackupCommand);
              command.setEnabled(isSingleDeviceSelected(that));
              command = mgrUtils.findCommand(that, RestoreCommand);
              command.setEnabled(isSingleDeviceSelected(that));
              //TODO: This should be easier, see smartTableHx:CommandEnablePolicy.singleSelection
            });
          });
      });
    };

  /**
   * Return the commands for the manager. This includes the basic commands
   * for a device manager with discovery support, and also adds the BACnet Aws
   * specific commands.
   *
   * @returns {Array.<module:bajaux/commands/Command>}
   */
  BacnetAwsDeviceUxManager.prototype.makeCommands = function () {
    var cmds = BacnetOwsDeviceUxManager.prototype.makeCommands.apply(this, arguments);
    return cmds.concat([new DeviceCommunicationControlCommand(this), new ReinitializeDeviceCommand(this), new BackupCommand(this), new RestoreCommand(this)]);
  };

  return (BacnetAwsDeviceUxManager);
});
