/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['Promise',
        'underscore',
        'nmodule/js/rc/asyncUtils/asyncUtils',
        'nmodule/js/rc/csrf/csrfUtil'], function (
         Promise,
         _,
         asyncUtils,
         csrfUtil) {
  'use strict';
  
  var doRequire = asyncUtils.doRequire,
    getCsrfToken = csrfUtil.getCsrfToken;
  
  var getBaja = _.once(function () { return doRequire('baja!'); });
  var getJQuery = _.once(function () { return doRequire('jquery'); });

  /**
   * @typedef {Object} module:nmodule/js/rc/rpc/rpc~RpcParams
   * @property {string} [ord] the ORD to send the RPC to
   * @property {string} [typeSpec] or, if a static RPC to a Type, that type spec
   * @property {string} methodName
   * @property {Array.<*>} args
   */
  
  /**
   * API Status: **Private**
   * @exports nmodule/js/rc/rpc/rpc
   */
  var exports = {};

  /**
   * Perform an RPC call using BajaScript.
   * 
   * @private
   * @param {module:nmodule/js/rc/rpc/rpc~RpcParams} arg RPC params, or to
   * perform multiple RPC calls at once, an array of param objects
   * @returns {Promise.<*|Array>} if a single param was given, the result of
   * that RPC call; if an array of param objects given, an array of results
   */
  exports.baja = function (arg) {
    return handleArray(arg, function (arr) {
      return getBaja()
        .then(function (baja) {
          var batch = new baja.comm.Batch();
          return batch.commit(Promise.all(arr.map(function (params) {
            return baja.rpc(_.extend({ batch: batch }, params));
          })));
        });
    });
  };
  
  exports.$ajax = function (arg) {
    return getJQuery()
      .then(function ($) {
        return $.ajax('/rpc', arg);
      });
  };

  /**
   * Perform an RPC call using AJAX.
   * 
   * @private
   * @param {module:nmodule/js/rc/rpc/rpc~RpcParams} arg RPC params, or to
   * perform multiple RPC calls at once, an array of param objects
   * @returns {Promise.<*|Array>} if a single param was given, the result of
   * that RPC call; if an array of param objects given, an array of results
   */
  exports.ajax = function (arg) {
    //TODO: can we do this without jQuery
    return handleArray(arg, function (arr) {
      return exports.$ajax({
        accepts: { json: "application/json" },
        dataType: "json",
        data: JSON.stringify(arr.map(toAjaxParams)),
        headers: { 'x-niagara-csrfToken': getCsrfToken() },
        type: 'POST'
      });
    });
  };

  /**
   * Perform an RPC call using whatever mechanism is configured and available.
   * 
   * @param {module:nmodule/js/rc/rpc/rpc~RpcParams} arg RPC params, or to
   * perform multiple RPC calls at once, an array of param objects
   * @returns {Promise.<*|Array>} if a single param was given, the result of
   * that RPC call; if an array of param objects given, an array of results
   */
  exports.rpc = function (arg) {
    if (require.specified('baja')) {
      return exports.baja(arg);
    } else if (require.specified('jquery')) {
      return exports.ajax(arg);
    } else {
      return Promise.reject(new Error('either BajaScript or jQuery required'));
    }
  };

  function toAjaxParams(params) {
    if (params.typeSpec) {
      return _.extend({ ord: 'type:' + params.typeSpec },
        _.omit(params, 'typeSpec'));
    } else {
      return params;
    }
  }

  function handleArray(arg, doHandleArray) {
    var isArray = Array.isArray(arg),
      arr = isArray ? arg : [ arg ];
    return doHandleArray(arr)
      .then(function (results) {
        return isArray ? results : results[0];
      });
  }
  
  return exports;
});
