/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['nmodule/js/rc/log/Log'], function (
         Log) {
  
  'use strict';

  /**
   * RequireJS plugin for retrieving Log instances. See {@link module:nmodule/js/rc/log/Log|Log}
   * for more details on working with logs.
   * 
   * @module log
   * @see module:nmodule/js/rc/log/Log
   * @see module:nmodule/js/rc/log/Log~Handler
   *
   * @example
   * <caption>Pass the plugin a logger name to get a preconfigured Log instance
   * for that name.</caption>
   * require(['log!nmodule.myModule.rc.views.FooView'], function (fooLog) {
   *   fooLog.info('info msg');
   *   fooLog.warning('warning msg');
   *   if (fooLog.isLoggable('FINE')) {
   *     fooLog.fine(buildExpensiveLogMessage());
   *   }
   * });
   *
   * @example
   * <caption>Pass the plugin "console" or "browser.console" to get Log itself,
   * whose API matches the browser console.</caption>
   * require(['log!console'], function (console) {
   *   console.log('this will log to the browser console',
   *     'and up to the station too');
   * });
   *
   * @example
   * <caption>Register custom log handlers using the logHandlers config. See
   * web:IRequireJsConfig for one way to accomplish this.</caption>
   * require.config['nmodule/js/rc/log/Log'].logHandlers = [
   *   'nmodule/myModule/rc/myCustomHandler'
   * ];
   *
   * // in myModule/src/rc/myCustomHandler.js:
   * define([], function () {
   *   'use strict';
   *   return {
   *     publish: function (name, level, msg) {
   *       //implement behavior here.
   *       //return writeToLogFile(name, level, msg);
   *       //return appendToHistory(name, level, msg);
   *       //return postToCloud(name, level, msg);
   *     }
   *   };
   * });
   * 
   * @example
   * <caption>In most cases, log levels are automatically set for you, and they
   * will match the settings you configured in the Logger Configuration
   * Workbench tool. But if you are building a RequireJS config from scratch,
   * you can specify log levels per-package using the logLevels config.</caption>
   * require.config['nmodule/js/rc/log/Log'].logLevels = {
   *   'my.package.name': 'FINE',
   *   'my.other.package.name': 'SEVERE'
   * };
   */
  return {
    load: function (name, req, onLoad) {
      switch (name) {
        case 'console':
        case 'browser.console':
          return onLoad(Log);
      }
      Log.getLogger(name)
        .then(onLoad)
        .catch(onLoad.error);
    }
  };
});