/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!',
        'lex!bacnet',
        'underscore',
        'nmodule/bacnet/rc/baja/datatypes/BacnetAddress',
        'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyPathMgrColumn',
        'nmodule/webEditors/rc/wb/table/model/Column',
        'nmodule/webEditors/rc/wb/table/model/columns/IconColumn',
        'nmodule/webEditors/rc/wb/table/model/columns/PropertyColumn',
        'nmodule/webEditors/rc/wb/table/tree/TreeTableModel'], function (
         baja,
         lexs,
         _,
         BacnetAddress,
         PropertyPathMgrColumn,
         Column,
         IconColumn,
         PropertyColumn,
         TreeTableModel) {

  'use strict';

  var bacnetLex = lexs[0],
      DISCOVERY_NODE_ICON = baja.Icon.make(['module://icons/x16/device.png']),
      UNSEEN = Column.flags.UNSEEN;

  function lex(key) {
    return bacnetLex.get(key);
  }

  //
  // Column for the row's icon - always returns the device icon.
  //

  var DiscoveryIconColumn = function () {
    IconColumn.apply(this, arguments);
  };
  DiscoveryIconColumn.prototype = Object.create(IconColumn.prototype);
  DiscoveryIconColumn.prototype.constructor = DiscoveryIconColumn;

  DiscoveryIconColumn.prototype.getValueFor = _.constant(DISCOVERY_NODE_ICON);

  //
  // Column for the MAC address. Note that this is a distinct
  // type from the MAC column type used for the main table.
  //

  var MacColumn = function (name, params) {
    Column.apply(this, arguments);
  };
  MacColumn.prototype = Object.create(Column.prototype);
  MacColumn.prototype.constructor = MacColumn;

  MacColumn.prototype.getValueFor = function (row) {
    var address = row.getSubject().get('address');

    return BacnetAddress.bytesToString(address.get('addressType').valueOf(),
      address.get('macAddress').getBytes());
  };

  //
  // Column for the size of the device's object list.
  //

  var SizeColumn = function (name, params) {
    Column.apply(this, arguments);
  };
  SizeColumn.prototype = Object.create(Column.prototype);
  SizeColumn.prototype.constructor = SizeColumn;

  SizeColumn.prototype.getValueFor = function (row) {
    var size = row.getSubject().get('listSize').valueOf();
    return size > 0 ? String(size) : lex('deviceManager.unknown');
  };

  var COLUMNS = [
    new DiscoveryIconColumn(),
    new PropertyColumn('deviceName', { displayName: lex('deviceManager.name') }),
    new PropertyColumn('objectId', { displayName: lex('deviceManager.deviceId') }),
    new PropertyPathMgrColumn('address/networkNumber', { displayName: lex('deviceManager.network') }),
    new MacColumn('address', { displayName: lex('deviceManager.mac') }),
    new PropertyColumn('vendorName', { displayName: lex('deviceManager.vendorName') }),
    new PropertyColumn('modelName', { displayName: lex('deviceManager.modelName') }),
    new PropertyColumn('maxApduLengthAccepted', { displayName: lex('deviceManager.maxApdu'), flags: UNSEEN }),
    new PropertyColumn('segmentationSupported', { displayName: lex('deviceManager.segmentation'), flags: UNSEEN }),
    new PropertyColumn('vendorId', { displayName: lex('deviceManager.vId'), flags: UNSEEN }),
    new SizeColumn('listSize', { displayName: lex('deviceManager.size') }),
    new PropertyColumn('encoding', { displayName: lex('deviceManager.charSet'), flags: UNSEEN }),
    new PropertyColumn('firmwareRevision', { displayName: lex('deviceManager.fwRev'), flags: UNSEEN }),
    new PropertyColumn('applicationSoftwareVersion', { displayName: lex('deviceManager.appSwVer'), flags: UNSEEN }),
    new PropertyColumn('duplicate', { displayName: lex('deviceManager.dup'), flags: UNSEEN })
  ];

  /**
   * API Status: **Private**
   * @exports nmodule/bacnet/rc/wb/mgr/model/BacnetDeviceLearnModel
   */
  var exports = {};

  /**
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel>}
   */
  exports.make = function () {
    return TreeTableModel.make({ columns: COLUMNS });
  };

  return (exports);
});
