/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/wb/mgr/commands/TimeSynchCommand
 */
define(['baja!',
        'lex!bacnet',
        'jquery',
        'Promise',
        'underscore',
        'dialogs',
        'bajaux/commands/Command',
        'nmodule/webEditors/rc/fe/feDialogs',
        'nmodule/webEditors/rc/wb/PropertySheet',
        'baja!bacnet:TimeSynchConfig,bacnet:TimeSynchJob,bacnet:BacnetNetwork'], function (
        baja,
        lexs,
        $,
        Promise,
        _,
        dialogs,
        Command,
        feDialogs,
        PropertySheet) {

  'use strict';

  var bacnetLex = lexs[0];

  /**
   * Return an object to be displayed in a property sheet for the time synch config.
   * This will use a couple of dynamic enums, which will be converted to a instance
   * of BTimeSynchConfig later.
   *
   * @returns {Promise<baja.Component>}
   */
  function makeConfigObject () {
    var timeSyncTypeEnum = baja.DynamicEnum.make({
      range: baja.EnumRange.make({
        ordinals: [0, 1],
        tags: [ baja.SlotPath.escape(bacnetLex.get('timeSynchType.local')),
                baja.SlotPath.escape(bacnetLex.get('timeSynchType.utc')) ]
      })
    });

    var addressRangeEnum = baja.DynamicEnum.make({
      range: baja.EnumRange.make({
        ordinals: [0, 1],
        tags: [ baja.SlotPath.escape(bacnetLex.get('timeSynchRange.local')),
                baja.SlotPath.escape(bacnetLex.get('timeSynchRange.global')) ]
      })
    });

    var config = baja.$('baja:Component', {
      timeSynchType: timeSyncTypeEnum,
      addressRange: addressRangeEnum
    });

    return config.add({
      slot: 'displayNames',
      value: baja.NameMap.make({
        timeSynchType: bacnetLex.get('timeSynchType.title'),
        addressRange: bacnetLex.get('timeSynchRange.title')
      }),
      flags: baja.Flags.HIDDEN
    })
    .then(function () { return config; });
  }

  /**
   * Show the given configuration object in a property sheet. The
   * values will be used to set the argument for the job.
   *
   * @param {baja.Component} config
   * @returns {Promise}
   */
  function showConfigDialog (config) {

    return feDialogs.showFor({
      title: bacnetLex.get('timeSynch.title'),
      type: PropertySheet,
      value: config,
      properties: {
        showHeader: false,
        showFooter: false
      }
    });
  }

  /**
   * Display an error dialog, inserting the error text into the failure message.
   */
  function showErrorDialog (err) {
    dialogs.showOk({
      title: bacnetLex.get('timeSynch.title'),
      content: bacnetLex.getSafe('timeSynch.fail').replace('{0}', err || '')
    });
  }

  /**
   * Command for invoking a time synchronization job on the BACnet network.
   *
   * @class
   * @alias module:nmodule/bacnet/rc/wb/mgr/commands/TimeSynchCommand
   * @extends module:bajaux/commands/Command
   */
  var TimeSynchCommand = function TimeSynchCommand(mgr) {
    var that = this;

    Command.call(this, {
      displayName: bacnetLex.get('timeSynch.label'),
      description: bacnetLex.get('timeSynch.description'),
      icon: bacnetLex.get('timeSynch.icon'),
      enabled: true,

      /**
       * Display a dialog with the choices of local/UTC time and local or remote
       * networks. Once that's confirmed, we invoke the command to start the job
       * and then set the job on the manager view.
       *
       * @alias module:nmodule/bacnet/rc/wb/mgr/commands/TimeSynchCommand#invoke
       * @returns {Promise}
       */
      func: function () {
        var network = mgr.getNetwork();

        dialogs.showYesNo({
          title: bacnetLex.get('timeSynch.title'),
          content: bacnetLex.getSafe('timeSynch.confirm'),
          yes: function () {
            var config;
            makeConfigObject()
              .then(function (c) {
                config = c;
                return showConfigDialog(config);
              })
              .then(function (diff) {

                if (diff) {

                  // Get the values from the dynamic enums we created for the config
                  // object and use them to set the properties of a TimeSynchConfig.
                  // This is then passed to the action to start the job.

                  var jobConfig = baja.$('bacnet:TimeSynchConfig', {
                    timeSynchType: !!config.get('timeSynchType').getOrdinal(),
                    addressRange: !!config.get('addressRange').getOrdinal()
                  });

                  return network.submitDeviceManagerJob(jobConfig)
                    .then(function (ord) {

                      // Pass the job to the manager. This will set it on the job bar and
                      // will listen for the completion event. It will check the type and
                      // display a message when complete.

                      mgr.setJob(baja.Ord.make({
                        base: baja.Ord.make('station:'),
                        child: ord.relativizeToSession()
                       }));
                     });
                }
                else {
                  return null;
                }
              })
              .catch(showErrorDialog);
          }
        });
      }
    });

    mgr.on('jobcomplete', function (job) {
      if (job.getType().is('bacnet:TimeSynchJob')) {
        that.complete(job);
      }
    });
  };

  TimeSynchCommand.prototype = Object.create(Command.prototype);
  TimeSynchCommand.prototype.constructor = TimeSynchCommand;

  /**
   * Function called when the time synch job has completed.
   *
   * @param {baja.Component} job - the time synch job, submitted by the invoke function
   * above.
   */
  TimeSynchCommand.prototype.complete = function (job) {
    var msg, cause;

    if ('success' === job.getJobState().getTag()) {
      msg = bacnetLex.getSafe('timeSynch.success');
    }
    else {
      cause = job.get('failureCause') || '';
      msg = bacnetLex.getSafe({ key: 'timeSynch.fail', args: [ cause ]});
    }

    dialogs.showOk({
      title: bacnetLex.getSafe('timeSynch.title'),
      content: msg
    });
  };

  return TimeSynchCommand;
});
