/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/util/ObjectTypeList
 */
define(['baja!',
        'jquery',
        'Promise',
        'underscore',
        'nmodule/bacnet/rc/util/BacnetConst',
        'nmodule/bacnet/rc/util/PropertyInfo'], function (
        baja,
        $,
        Promise,
        _,
        BacnetConst,
        PropertyInfo) {

  'use strict';

  var instances = [],

      DEFAULT_INSTANCE_KEY = '__default__', // key used for either one of the two ords below

      DEFAULT_ORD  = 'file:!defaults/bacnetObjectTypes.xml',
      FALLBACK_ORD = 'module://bacnet/com/tridium/bacnet/objectTypes.xml';

  /**
   * ObjectTypeList provides support for mapping metadata information about the
   * various standard BACnet object types.  Proprietary object types, or proprietary
   * properties added to standard object types, are not handled by the ObjectTypeList.
   *
   * @class
   * @alias module:nmodule/bacnet/rc/util/ObjectTypeList
   */
  var ObjectTypeList = function ObjectTypeList() {
  };

  /**
   * Static `make` function to obtain an `ObjectTypeList`. This can be called
   * with no arguments to obtain the generic vendor neutral type list, or can
   * be supplied with an ord argument to load a type list for a specific vendor's
   * device. This ord is expected to be specified as a dynamic property on a device.
   *
   * @static
   *
   * @param {Object} [params] - optional parameter object.
   * @param {String} [params.ord] - optional ord allowing a vendor specific list to be obtained.
   *
   * @returns {module:nmodule/bacnet/rc/util/ObjectTypeList}
   */
  ObjectTypeList.make = function (params) {
    var ord = params && params.ord,
        key = ord ? String(ord) : DEFAULT_INSTANCE_KEY;

    if (instances[key]) {
      return Promise.resolve(instances[key]);
    }

    var typeList = new ObjectTypeList();

    return loadObjectTypeData(ord)
      .then(function (objects) {
        typeList.$objects = objects;
        instances[key] = typeList;
        return typeList;
      });
  };

  ////////////////////////////////////////////////////////////////
  // Access
  ////////////////////////////////////////////////////////////////

  /**
   * Find information about a property, from an object type and a property ID.
   *
   * @param {number} objectType
   * @param {number} propertyId
   *
   * @returns {module:nmodule/bacnet/rc/util/PropertyInfo}
   */
  ObjectTypeList.prototype.getPropertyInfo = function (objectType, propertyId) {
    var obj = this.$objects[objectType], prop;

    prop = obj && obj.$properties[propertyId];
    return prop ? prop : null;
  };

  /**
   * Test whether the given objectType number is known. Returns `true` if the
   * list contains an entry for this type.
   *
   * @param {number} objectType
   * @returns {boolean}
   */
  ObjectTypeList.prototype.isObjectTypeKnown = function (objectType) {
    return !!(this.$objects[objectType]);
  };

  ////////////////////////////////////////////////////////////////
  // XML Loading
  ////////////////////////////////////////////////////////////////

  function attr (elem, name) {
    return elem.getAttribute(name);
  }

  /**
   * Return the contents of the given file descriptor by reading it from the station
   * via HTTP/S.
   *
   * @param {baja.file.File} file
   * @returns {Promise.<string>}
   */
  function fetchOnline (file) {
    return Promise.resolve($.ajax(file.getReadUri() + '?snoop=false', { dataType: "text" }));
  }

  /**
   * Return the given file's contents by obtaining it via RPC if the view is offline.
   *
   * @param {baja.file.File} file
   * @returns {Promise.<string>}
   */
  function fetchOffline (file) {
    return baja.rpc("type:web:FileRpc", "readTextFile", file.getNavOrd().toString());
  }

  /**
   * Fetch the contents of the file - choosing to get it either online or offline.
   */
  function fetchContent (file) {
    return baja.isOffline() ? fetchOffline(file) : fetchOnline(file);
  }

  /**
   * Read in the content of the XML file and return the parsed contents.
   *
   * @param {baja.file.File} file - the XML file
   * @returns {Promise.<Array.<Object>}
   */
  function loadXml(file) {
    return fetchContent(file)
      .then(function (content) {
        var objects = parseObjectXml(content);
        return objects;
      });
  }

  /**
   * Load and parse the file for the given ord.
   *
   * @param {string} ord - the ord for the object type XML file.
   *
   * @returns {Promise.<module:nmodule/bacnet/rc/util/ObjectTypeList>}
   */
  function loadOrd (ord) {
    return baja.Ord.make(ord).get()
      .then(function (file) {
        return loadXml(file);
      });
  }

  /**
   * Try to get a generic, vendor neutral objectTypes.xml file. This will try
   * to get a version from the file system first and, if that fails, fall back to
   * the standard Tridium shipped one that is built as part of the bacnet module.
   *
   * @returns {module:nmodule/bacnet/rc/util/ObjectTypeList}
   */
  function loadGenericFile () {
    return loadOrd(DEFAULT_ORD, true)
      .catch(function () {
        return loadOrd(FALLBACK_ORD, true);
      });
  }

  /**
   * Load a specific file specified by an ord in the `make()` method paramters.
   *
   * @param {string} ord - an ord passed as the parameter.
   * @returns {module:nmodule/bacnet/rc/util/ObjectTypeList}
   */
  function loadFromOrdParam (ord) {
    return loadOrd(ord, false);
  }

  /**
   * Load the type data from the XML file referred to by the ord. If the ord
   * is not defined, it will try to load the standard geneneric file that is
   * not specific to any vendor or device.
   *
   * @param {string} ord - an ord passed as the parameter.
   * @returns {module:nmodule/bacnet/rc/util/ObjectTypeList}
   */
  function loadObjectTypeData (ord) {
    return ord ? loadFromOrdParam(ord) : loadGenericFile();
  }

  /**
   * Parse the 'Object' XML element and create `PropertyInfo` instances for each
   * 'Property' element we find.
   *
   * @param {Object} - an XML element object for the 'object' element.
   * @returns {Object} an object representing a map of `PropertyInfo` instances.
   */
  function parseXmlProperties (elem) {
    var props = {};

    $(elem).find('property').each(function (index, elem) {
      var info = PropertyInfo.$makeFromXmlElement(elem);
      props[info.getId()] = info;
    });

    return props;
  }

  /**
   * Parses the XML file and creates a map of object type ids to the object
   * descriptions.
   *
   * @param {String} content - contents of the object type XML file.
   * @returns {Object} - an object whose properties are the keyed object types.
   */
  function parseObjectXml (content) {
    var xml = $.parseXML(content),
        objects = {};

    $(xml).find('object').each(function (index, elem) {
      var objectType = {
        $name: attr(elem, 'n'),
        $typeId: attr(elem, 't'),
        $properties: parseXmlProperties(elem)
      };

      objects[objectType.$typeId] = objectType;
    });

    return objects;
  }

  return ObjectTypeList;
});
