/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/BacnetTimeStampEditor
 */
define([
  'baja!',
  'jquery',
  'underscore',
  'Promise',
  'nmodule/webEditors/rc/fe/baja/BaseEditor',
  'nmodule/webEditors/rc/fe/baja/DisplayOnlyEditor',
  'nmodule/webEditors/rc/fe/baja/util/compUtils',
  'nmodule/webEditors/rc/util/htmlUtils',
  'nmodule/bacnet/rc/fe/BacnetUnsignedEditor',
  'hbs!nmodule/bacnet/rc/fe/templates/BacnetTimeStampEditor',
  'bajaux/util/CommandButtonGroup',
  'bajaux/commands/ToggleCommandGroup',
  'nmodule/webEditors/rc/fe/fe',
  'bajaux/Widget',
  'bajaux/events',
  'bajaux/mixin/batchSaveMixin',
  'lex!bacnet',
  'nmodule/bacnet/rc/baja/util/BacnetDateUtil',
  'baja!bacnet:BacnetTime',
  'baja!bacnet:BacnetDate'
], function
  (baja,
   $,
   _,
   Promise,
   BaseEditor,
   DisplayOnlyEditor,
   compUtils,
   htmlUtils,
   BacnetUnsignedEditor,
   templateEditor,
   CommandButtonGroup,
   ToggleCommandGroup,
   fe,
   Widget,
   events,
   batchSaveMixin,
   lexs,
   BacnetDateUtil) {
  'use strict';

  var escapeHtml = htmlUtils.escapeHtml,
    bulkCopy = compUtils.bulkCopy,
    lex = lexs[0],
    TIME_CHOICE = 0,
    SEQ_CHOICE = 1,
    DATE_TIME_CHOICE = 2;

  var DESTROY_EVENT = events.DESTROY_EVENT,
    INITIALIZE_EVENT = events.INITIALIZE_EVENT,
    LOAD_EVENT = events.LOAD_EVENT,
    ENABLE_EVENT = events.ENABLE_EVENT,
    DISABLE_EVENT = events.DISABLE_EVENT,
    MODIFY_EVENT = events.MODIFY_EVENT,
    READONLY_EVENT = events.READONLY_EVENT,
    WRITABLE_EVENT = events.WRITABLE_EVENT;

  /**
   * A Bajaux editor for bacnet:BacnetTimeStamp type
   *
   * @class
   * @extends module:bajaux/Widget
   * @alias module:nmodule/bacnet/rc/fe/BacnetTimeStampEditor
   */
  var BacnetTimeStampEditor = function BacnetTimeStampEditor() {
    var that = this;
    BaseEditor.apply(that, arguments);

    if (that.getFormFactor() === Widget.formfactor.mini) {
      DisplayOnlyEditor.$mixin(that);
    } else {
      batchSaveMixin(that);
      that.$toggleCmdGrp = new ToggleCommandGroup({
        onChange: function (value) {
          that.$toggleDisplay(value);
        },
        commands: [
          {
            module: "bacnet",
            lex: "BacnetTS.commands.toggle.time",
            value: TIME_CHOICE
          },
          {
            module: "bacnet",
            lex: "BacnetTS.commands.toggle.seqNum",
            value: SEQ_CHOICE
          },
          {
            module: "bacnet",
            lex: "BacnetTS.commands.toggle.dateTime",
            value: DATE_TIME_CHOICE
          }
        ]
      });
    }

  };
  BacnetTimeStampEditor.prototype = Object.create(BaseEditor.prototype);
  BacnetTimeStampEditor.prototype.constructor = BacnetTimeStampEditor;

  BacnetTimeStampEditor.prototype.valueToString = function (value) {
    var dispVal = "";
    if (baja.hasType(value, 'bacnet:BacnetTimeStamp')) {
      var encVal = value.getDisplay();
      if (encVal !== null) {
        dispVal = encVal.substring(encVal.indexOf(':') + 1);
      } else {
        switch (value && value.getChoice()) {
          case TIME_CHOICE:
            dispVal = value.getTime().encodeToString();
            break;
          case SEQ_CHOICE:
            dispVal = value.getSequenceNumber().encodeToString();
            break;
          case DATE_TIME_CHOICE:
            var dt = value.getDateTime();
            dispVal = BacnetDateUtil.getDisplayBacnetDateString(dt.getDate().encodeToString()) + "_" + dt.getTime().encodeToString();
            break;
          default:
            break;
        }
      }
    } else {
      dispVal = value && value.getTime().encodeToString();
    }

    return escapeHtml(dispVal || "");
  };

  /**
   * @param {JQuery} dom
   * @returns {Promise}
   */
  BacnetTimeStampEditor.prototype.doInitialize = function (dom) {
    var that = this;

    //Disable events from bubbling up
    dom.on([
      DESTROY_EVENT,
      INITIALIZE_EVENT,
      LOAD_EVENT,
      ENABLE_EVENT,
      DISABLE_EVENT,
      READONLY_EVENT,
      WRITABLE_EVENT
    ].join(' '), '.editor', false);

    dom.on(MODIFY_EVENT, '.bajaux-initialized', function () {
      that.setModified(true);
      return false;
    });

    dom.html(templateEditor({
      labels: {
        time: lex.get('BacnetTS.editor.label.time'),
        seq: lex.get('BacnetTS.editor.label.seqNumber'),
        dt: lex.get('BacnetTS.editor.label.dateTime')
      }
    }));

    return Promise.join(
      fe.buildFor({
        dom: $('.toggleCommandGrp', dom),
        type: CommandButtonGroup,
        value: that.$getToggleCommandGroup(),
        initializeParams: {
          toggleGroup: true
        }
      }),
      fe.buildFor({
        value: 0,
        type: BacnetUnsignedEditor,
        dom: that.jq().find('.bacTS-seq'),
        properties: { prelabel: lex.get("BacnetTS.editor.label.seqNumber") + ":", precision: 0 },
        formFactor: 'mini'
      }),
      fe.buildFor({
        value: baja.$("bacnet:BacnetTime"),
        dom: that.jq().find('.bts-time'),
        properties: { prelabel: lex.get("BacnetTS.editor.label.time") },
        formFactor: 'mini'
      }),
      fe.buildFor({
        value: baja.$("bacnet:BacnetDate"),
        dom: that.jq().find('.bacTS-dtd'),
        formFactor: 'mini'
      }),
      fe.buildFor({
        value: baja.$("bacnet:BacnetTime"),
        dom: that.jq().find('.bacTS-dtt'),
        formFactor: 'mini'
      })
    );
  };

  BacnetTimeStampEditor.prototype.doLoad = function (value) {
    var that = this;
    var choice = value && value.getChoice();
    that.$toggleDisplay(choice);
    switch (choice) {
      case TIME_CHOICE:
        that.$getTimeToggleCommand().setSelected(true);
        return that.$loadTime(value.getTime());
      case SEQ_CHOICE:
        that.$getSeqToggleCommand().setSelected(true);
        return that.$loadSeqNum(value.getSequenceNumber());
      case DATE_TIME_CHOICE:
        that.$getDtTmToggleCommand().setSelected(true);
        return that.$loadDateTime(value.getDateTime());
      default:
        break;
    }
  };

  BacnetTimeStampEditor.prototype.$loadTime = function (timeValue) {
    return this.jq().find('.bts-time').data('widget').load(timeValue || baja.$("bacnet:BacnetTime"));
  };
  BacnetTimeStampEditor.prototype.$loadSeqNum = function (seqNumValue) {
    return this.jq().find('.bacTS-seq').data('widget').load(seqNumValue || 0);
  };
  BacnetTimeStampEditor.prototype.$loadDateTime = function (datetimeValue) {
    var dtValue = datetimeValue && datetimeValue.getDate();
    var tValue = datetimeValue && datetimeValue.getTime();
    return Promise.join(
      this.jq().find('.bacTS-dtd').data('widget').load(dtValue || baja.$("bacnet:BacnetDate")),
      this.jq().find('.bacTS-dtt').data('widget').load(tValue || baja.$("bacnet:BacnetTime"))
    );
  };

  BacnetTimeStampEditor.prototype.doRead = function () {
    var that = this;
    var choice = parseInt(that.$getToggleCommandGroup().getSelectedValue(), 10);
    return this.getChildEditors().readAll().then(function (val) {
      var ts = baja.$("bacnet:BacnetTimeStamp");

      switch (choice) {
        case TIME_CHOICE:
          ts.setTime(val[0]);
          ts.setChoice(0);
          break;
        case SEQ_CHOICE:
          ts.setSequenceNumber(val[1]);
          ts.setChoice(1);
          break;
        case DATE_TIME_CHOICE:
          ts.getDateTime().setTime(val[3]);
          ts.getDateTime().setDate(val[2]);
          ts.setChoice(2);
          break;
      }

      return ts;
    });
  };

  BacnetTimeStampEditor.prototype.doSave = function (readValue, params) {
    var batch = params && params.batch,
        progressCallback = params && progressCallback,
        value = this.value(),
        source = readValue,
        target = value,
        slots,
        prom;

    switch (readValue.get('choice').valueOf()) {
      case TIME_CHOICE:
        slots = [ 'time' ];
        break;
      case SEQ_CHOICE:
        slots = [ 'sequenceNumber' ];
        break;
      case DATE_TIME_CHOICE:
        source = readValue.get('dateTime');
        target = value.get('dateTime');
        slots = [ 'date', 'time' ];
        break;
    }

    prom = Promise.join(
      bulkCopy(source, target, slots, batch),
      bulkCopy(readValue, value, [ 'choice' ], batch)
    );
    if (progressCallback) { progressCallback(batchSaveMixin.COMMIT_READY); }
    return prom;
  };

  BacnetTimeStampEditor.prototype.$getToggleCommandGroup = function () {
    return this.$toggleCmdGrp;
  };

  BacnetTimeStampEditor.prototype.$getSeqToggleCommand = function () {
    return this.$getToggleCommandGroup().get(1);
  };

  BacnetTimeStampEditor.prototype.$getTimeToggleCommand = function () {
    return this.$getToggleCommandGroup().get(0);
  };

  BacnetTimeStampEditor.prototype.$getDtTmToggleCommand = function () {
    return this.$getToggleCommandGroup().get(2);
  };

  BacnetTimeStampEditor.prototype.doDestroy = function () {
    return this.getChildWidgets().destroyAll();
  };

  /**
   * Set all child editors readonly/writable.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  BacnetTimeStampEditor.prototype.doReadonly = function (readonly) {
    return this.getChildWidgets().setAllReadonly(readonly);
  };

  /**
   * Set all child editors enabled/disabled.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  BacnetTimeStampEditor.prototype.doEnabled = function (enabled) {
    return this.getChildWidgets().setAllEnabled(enabled);
  };

  /**
   * change the display state based on the time stamp choice
   * @private
   * @param {Number} choice - The time stamp value type [0=time, 1=seq num, 2=date and time]
   */
  BacnetTimeStampEditor.prototype.$toggleDisplay = function (choice) {
    var seqCtrl = this.jq().find(".bacTS-seq");
    var timeCtrl = this.jq().find(".bacTS-time");
    var dtTmCtrl = this.jq().find(".date-time");
    switch (choice) {
      case 0:
        seqCtrl.hide();
        dtTmCtrl.hide();
        timeCtrl.show();
        break;
      case 1:
        timeCtrl.hide();
        dtTmCtrl.hide();
        seqCtrl.show();
        break;
      case 2:
        timeCtrl.hide();
        seqCtrl.hide();
        dtTmCtrl.show();
        break;
      default:
        break;
    }
  };

  return (BacnetTimeStampEditor);
});
