/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/BacnetScaleEditor
 */
define(['baja!',
        'lex!bacnet',
        'bajaux/events',
        'bajaux/mixin/batchSaveMixin',
        'Promise',
        'nmodule/webEditors/rc/fe/fe',
        'nmodule/webEditors/rc/fe/baja/BaseEditor',
        'nmodule/webEditors/rc/fe/baja/OrderedMapEditor',
        'nmodule/webEditors/rc/fe/baja/util/compUtils',
        'css!nmodule/bacnet/rc/bacnet'], function (
         baja,
         lexs,
         events,
         batchSaveMixin,
         Promise,
         fe,
         BaseEditor,
         OrderedMapEditor,
         compUtils) {

  'use strict';

  var bacnetLex = lexs[0],

      bulkCopy = compUtils.bulkCopy,

      MODIFY_EVENT = events.MODIFY_EVENT,
      FLOAT_CHOICE = 0,
      INT_CHOICE = 1;

  var CHOICES = (function () {
    var map = new baja.OrderedMap();
    map.put(bacnetLex.get({ key: 'bacnetScale.float', def: 'Float' }), FLOAT_CHOICE);
    map.put(bacnetLex.get({ key: 'bacnetScale.int', def: 'Int' }), INT_CHOICE);
    return map;
  }());

  /**
   * Field editor for working with `bacnet:BacnetScale` instances.
   *
   * @class
   * @alias module:nmodule/bacnet/rc/fe/BacnetScaleEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   */
  var BacnetScaleEditor = function BacnetScaleEditor() {
    BaseEditor.apply(this, arguments);
    batchSaveMixin(this);
  };
  BacnetScaleEditor.prototype = Object.create(BaseEditor.prototype);
  BacnetScaleEditor.prototype.constructor = BacnetScaleEditor;

  /**
   * Set event handlers to rebuild the `scale` editor as either a double or
   * integer editor, depending on which choice is selected.
   * @param {JQuery} dom
   */
  BacnetScaleEditor.prototype.doInitialize = function (dom) {
    var that = this;

    dom.html('<div class="js-choice"/><div class="js-scale"/>');

    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });

    dom.on(MODIFY_EVENT, '.js-choice', function () {
      var choice = that.$getChoiceEditor().getSelectedValue(),
          scaleEd = that.$getScaleEditor();

      scaleEd.read()
        .then(function (scale) {
          return that.$loadScale(choice, scale);
        })
        .catch(baja.error);
    });

    return fe.buildFor({
      dom: dom.children('.js-choice'),
      type: OrderedMapEditor,
      value: CHOICES,
      formFactor: 'mini'
    });
  };


  /**
   * @param {baja.Struct} bacnetScale the `bacnet:BacnetScale` to load
   * @returns {Promise}
   */
  BacnetScaleEditor.prototype.doLoad = function (bacnetScale) {
    return this.$loadScale(bacnetScale.get('choice'), bacnetScale.get('scale'));
  };

  /**
   * Read a new `BacnetScale` with currently selected values.
   * @returns {Promise.<baja.Struct>}
   */
  BacnetScaleEditor.prototype.doRead = function () {
    return this.getChildEditors().readAll()
      .spread(function (choice, scale) {
        return baja.$('bacnet:BacnetScale', { choice: choice, scale: scale });
      });
  };

  /**
   * Apply the currently entered `choice` and `scale` values back to the
   * `BacnetScale`. Respects the `batchSaveMixin` contract.
   *
   * @param {baja.Struct} bacnetScale a `bacnet:BacnetScale` to save
   * @param {Object} params
   */
  BacnetScaleEditor.prototype.doSave = function (bacnetScale, params) {
    var batch = params && params.batch,
        progressCallback = params && params.progressCallback;

    var prom = bulkCopy(bacnetScale, this.value(), [ 'choice', 'scale' ], batch);
    if (progressCallback) { progressCallback(batchSaveMixin.COMMIT_READY); }
    return prom;
  };

  /**
   * Set all child editors enabled/disabled.
   * @param {boolean} enabled
   * @returns {Promise}
   */
  BacnetScaleEditor.prototype.doEnabled = function (enabled) {
    return this.getChildEditors().setAllEnabled(enabled);
  };

  /**
   * Set all child editors readonly/writable.
   * @param {boolean} readonly
   * @returns {Promise}
   */
  BacnetScaleEditor.prototype.doReadonly = function (readonly) {
    return this.getChildEditors().setAllReadonly(readonly);
  };

  /**
   * Destroy all child editors.
   * @returns {Promise}
   */
  BacnetScaleEditor.prototype.doDestroy = function () {
    return this.getChildEditors().destroyAll();
  };

  /**
   * Update the selected choice and scale values, rebuilding the `scale` editor
   * in the process as the choice may have changed.
   *
   * @private
   * @param {number|baja.Integer} choice
   * @param {number|baja.Integer} scale
   * @returns {Promise}
   */
  BacnetScaleEditor.prototype.$loadScale = function (choice, scale) {
    var that = this,
        scaleEd = that.$getScaleEditor();

    choice = choice.valueOf();
    scale = scale.valueOf();

    if (choice !== FLOAT_CHOICE) { scale = baja.Integer.make(scale); }

    return Promise.resolve(scaleEd && scaleEd.destroy())
      .then(function () {
        that.$getChoiceEditor().setSelectedValue(choice, true);
        return fe.buildFor({
          value: scale,
          dom: that.$getScaleElement(),
          formFactor: 'mini'
        });
      });
  };


  /**
   * Get the editor for the selected choice.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/OrderedMapEditor}
   */
  BacnetScaleEditor.prototype.$getChoiceEditor = function () {
    return this.jq().children('.js-choice').data('widget');
  };

  /**
   * Get the editor for the `scale` slot (this can contain either a
   * `baja:Double` or `baja:Integer` depending on selected choice).
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/NumericEditor}
   */
  BacnetScaleEditor.prototype.$getScaleEditor = function () {
    return this.$getScaleElement().data('widget');
  };

  /**
   * @private
   * @returns {JQuery} element containing the `scale` editor
   */
  BacnetScaleEditor.prototype.$getScaleElement = function () {
    return this.jq().children('.js-scale');
  };

  return BacnetScaleEditor;
});
