/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/BacnetPriorityValueEditor
 */
define([
  'baja!',
  'lex!bacnet',
  'bajaux/events',
  'bajaux/mixin/batchSaveMixin',
  'jquery',
  'Promise',
  'underscore',
  'nmodule/js/rc/switchboard/switchboard',
  'nmodule/webEditors/rc/fe/baja/BaseEditor',
  'nmodule/webEditors/rc/fe/baja/util/typeUtils',
  'nmodule/webEditors/rc/fe/fe',
  'nmodule/webEditors/rc/servlets/registry',
  'css!nmodule/bacnet/rc/bacnet'
], function (
  baja,
  lexs,
  events,
  batchSaveMixin,
  $,
  Promise,
  _,
  switchboard,
  BaseEditor,
  typeUtils,
  fe,
  registry
) {

  'use strict';

  var TYPE_MAP = {
    'bacnet:BacnetNull': 'NULL',
    'baja:Float': 'Real',
    'baja:Double': 'Double',
    'bacnet:BacnetBinaryPv': 'Binary',
    'bacnet:BacnetUnsigned': 'Integer',
    'baja:String': 'String'
  };

  /**
   * Editor for handling `bacnet:BacnetPriorityValue` values.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/bacnet/rc/fe/BacnetPriorityValueEditor
   */
  var BacnetPriorityValueEditor = function BacnetPriorityValueEditor() {
    BaseEditor.apply(this, arguments);
    batchSaveMixin(this);
    switchboard(this, {
      '$initSubEditor': { allow: 'oneAtATime', onRepeat: 'preempt' }
    });
  };
  BacnetPriorityValueEditor.prototype = Object.create(BaseEditor.prototype);
  BacnetPriorityValueEditor.prototype.constructor = BacnetPriorityValueEditor;

  /**
   * Create the DOM structure that contains the drop down box and the div
   * to hold the sub-editor.
   *
   * @param {JQuery} dom
   */
  BacnetPriorityValueEditor.prototype.doInitialize = function (dom) {
    var that = this,
        options = _.map(TYPE_MAP, function (value, key) {
          return $('<option class="ux-option"/>')
            .val(key).text(value);
        });
    dom.append('<select class="ux-select js-type"/>');
    dom.append('<div class="js-subEditor"/>');
    that.$getSelect().html(options);
    dom.on('change', 'select.js-type', function () {
      that.setModified(true);
      that.$initSubEditor().catch(baja.error);
      return false;
    });
    dom.on(events.MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });
  };

  /**
   * Make a sub-editor depending on which type is selected in the drop
   * down box and place it in the dom.
   *
   * @private
   * @param {JQuery} dom
   * @returns {Promise}
   */
  BacnetPriorityValueEditor.prototype.$initSubEditor = function () {
    var that = this,
        subEditor = that.$getSubEditor();
    return Promise.join(
      typeUtils.importTypes(that.$getSelect().val()),
      subEditor && subEditor.destroy()
    ).spread(function (type) {
      return fe.buildFor({
        value: baja.$(type),
        dom: $('.js-subEditor', that.jq()),
        formFactor: 'mini'
      });
    });
  };

  /**
   * Load a `bacnet:BacnetPriorityValue` into the editor.
   *
   * @param {baja.Complex} priorityValue
   * @returns {Promise}
   */
  BacnetPriorityValueEditor.prototype.doLoad = function (priorityValue) {
    var that = this,
        typeSpec = priorityValue.get('value').getType().getTypeSpec();
    if (!TYPE_MAP[typeSpec]) {
      throw new Error(
        'Cannot load BacnetPriorityValue, unsupported primitive type: ' +
        typeSpec);
    }
    that.$getSelect().val(typeSpec);
    return that.$initSubEditor()
      .then(function () {
        var subEditor = that.$getSubEditor();
        return subEditor && subEditor.load(priorityValue.get('value'));
      });
  };

  /**
   * Read a new `bacnet:BacnetPriorityValue` instance from the
   * sub-editor in the dom.
   *
   * @returns {Promise.<baja.Complex>}
   */
  BacnetPriorityValueEditor.prototype.doRead = function () {
    return this.$getSubEditor().read()
      .then(function (value) {
        return baja.$('bacnet:BacnetPriorityValue', { value: value });
      });
  };

  /**
   * Save a new `bacnet:BacnetPriorityValue`.
   *
   * @param {baja.Complex} validPv the validated `bacnet:BacnetPriorityValue`
   * to be saved.
   * @param {Object} [params] optional params object passed to `save()`.
   * @returns {Promise}
   */
  BacnetPriorityValueEditor.prototype.doSave = function (validPv, params) {
    var batch = params && params.batch,
        progressCallback = params && params.progressCallback,
        prom = this.value().set({
          slot: 'value', value: validPv.get('value'), batch: batch
        });

    if (progressCallback) { progressCallback(batchSaveMixin.COMMIT_READY); }

    return prom;
  };

  /**
   * Destroy all child editors.
   *
   * @returns {Promise}
   */
  BacnetPriorityValueEditor.prototype.doDestroy = function () {
    return this.getChildEditors().destroyAll();
  };

  /**
   * Set all child editors to enabled/disabled.
   *
   * @param {Boolean} enabled
   * @returns {Promise|undefined}
   */
  BacnetPriorityValueEditor.prototype.doEnabled = function (enabled) {
    this.$getSelect().prop('disabled', this.isReadonly() || !enabled);
    return this.$getSubEditor().setEnabled(enabled);
  };

  /**
   * Set all child editors to readonly/writable.
   *
   * @param {Boolean} readonly
   * @returns {Promise|undefined}
   */
  BacnetPriorityValueEditor.prototype.doReadonly = function (readonly) {
    this.$getSelect().prop('disabled', !this.isEnabled() || readonly);
    return this.$getSubEditor().setReadonly(readonly);
  };

  /**
   * Get the sub-editor for the selected type.
   *
   * @private
   * @returns {*}
   */
  BacnetPriorityValueEditor.prototype.$getSubEditor = function () {
    return this.jq().children('.js-subEditor').data('widget');
  };

  /**
   * Get the select element.
   *
   * @private
   * @returns {JQuery}
   */
  BacnetPriorityValueEditor.prototype.$getSelect = function () {
    return this.jq().children('select');
  };

  return BacnetPriorityValueEditor;
});
