/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/BacnetObjectIdEditor
 */
define([
  'baja!',
  'jquery',
  'nmodule/webEditors/rc/fe/baja/BaseEditor',
  'underscore',
  'Promise',
  'nmodule/webEditors/rc/fe/fe',
  'nmodule/bacnet/rc/baja/datatypes/BacnetObjectIdentifier',
  'bajaux/events',
  'baja!bacnet:BacnetObjectType'], function
  (baja,
   $,
   BaseEditor,
   _,
   Promise,
   fe,
   BacnetObjectIdentifier,
   events) {

  'use strict';

  var DESTROY_EVENT = events.DESTROY_EVENT,
    INITIALIZE_EVENT = events.INITIALIZE_EVENT,
    LOAD_EVENT = events.LOAD_EVENT,
    ENABLE_EVENT = events.ENABLE_EVENT,
    DISABLE_EVENT = events.DISABLE_EVENT,
    MODIFY_EVENT = events.MODIFY_EVENT,
    READONLY_EVENT = events.READONLY_EVENT,
    WRITABLE_EVENT = events.WRITABLE_EVENT;

  /**
   * Editor for handling `bacnet:BacnetObjectIdentifier` values.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/bacnet/rc/fe/BacnetObjectIdEditor
   */
  var BacnetObjectIdEditor = function BacnetObjectIdEditor() {
    BaseEditor.apply(this, arguments);
  };
  BacnetObjectIdEditor.prototype = Object.create(BaseEditor.prototype);
  BacnetObjectIdEditor.prototype.constructor = BacnetObjectIdEditor;

  /**
   * Initialize the bacnet object id editor
   * @param {JQuery} dom
   */
  BacnetObjectIdEditor.prototype.doInitialize = function (dom) {
    var that = this;
    //Disable events from bubbling up
    dom.on([
      DESTROY_EVENT,
      INITIALIZE_EVENT,
      LOAD_EVENT,
      ENABLE_EVENT,
      DISABLE_EVENT,
      READONLY_EVENT,
      WRITABLE_EVENT
    ].join(' '), '.editor', false);

    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });

    return Promise.join(
      fe.buildFor({
        dom: $('<span class="js-objType" />').appendTo(dom),
        value: baja.DynamicEnum.make({
          range: baja.$('bacnet:BacnetObjectType').getRange()
        }),
        formFactor: 'mini'
      }),
      fe.buildFor({
        dom: $('<span class="js-instance" />').appendTo(dom),
        value: baja.Integer.DEFAULT,
        formFactor: 'mini'
      }));
  };

  /**
   * @param {baja.Simple}  boValue `bacnet:BacnetObjectIdentifier` value to load
   */
  BacnetObjectIdEditor.prototype.doLoad = function (boValue) {
    var that = this,
      objType;
    var range = this.properties().getValue('range');
    if (range) {
      objType = baja.DynamicEnum.make({
        ordinal: boValue.getObjectType(),
        range: range
      });
    } else {
      objType = baja.DynamicEnum.make({
        ordinal: boValue.getObjectType(),
        range: baja.$('bacnet:BacnetObjectType').getRange()
      });
    }
    return Promise.join(
      that.$getSelect().load(
        objType
      ),
      that.$getInstanceInput().load(boValue.getInstanceNumber())
    );
  };
  /**
   * Read the set values in UI
   * @return {baja.Simple} The read `bacnet:BacnetObjectIdentifier` value
   */
  BacnetObjectIdEditor.prototype.doRead = function () {
    return Promise.join(
      this.$getSelect().read(),
      this.$getInstanceInput().read()).spread(function (objType, objInst) {
        return baja.$("bacnet:BacnetObjectIdentifier", objType.getOrdinal(), objInst);
      });
  };

  /**
   * Make the widgets read only
   *
   * @param {Boolean} readonly
   */
  BacnetObjectIdEditor.prototype.doReadonly = function (readonly) {
    return Promise.join(
      this.$getSelect().setReadonly(readonly),
      this.$getInstanceInput().setReadonly(readonly)
    );
  };

  /**
   * Make the widgets enabled/disabled
   *
   * @param {Boolean} enabled
   */
  BacnetObjectIdEditor.prototype.doEnabled = function (enabled) {
    return Promise.join(
      this.$getSelect().setEnabled(enabled),
      this.$getInstanceInput().setEnabled(enabled)
    );
  };


  /**
   * Destroy all the child editors
   * @returns {*}
   */
  BacnetObjectIdEditor.prototype.doDestroy = function () {
    return this.getChildWidgets().destroyAll();
  };

  /**
   * Get the select element.
   *
   * @private
   * @returns {JQuery}
   */
  BacnetObjectIdEditor.prototype.$getSelect = function () {
    return this.jq().children('.js-objType').data('widget');
  };

  /**
   * Get the instance element.
   *
   * @private
   * @returns {JQuery}
   */
  BacnetObjectIdEditor.prototype.$getInstanceInput = function () {
    return this.jq().children('.js-instance').data('widget');
  };

  return BacnetObjectIdEditor;
});
