/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/BacnetAnyEditor
 */
define(['baja!',
        'bajaux/events',
        'bajaux/mixin/batchSaveMixin',
        'bajaux/mixin/subscriberMixIn',
        'lex!bacnet',
        'nmodule/bacnet/rc/baja/datatypes/BacnetAny',
        'nmodule/webEditors/rc/fe/fe',
        'nmodule/webEditors/rc/fe/baja/BaseEditor',
        'nmodule/webEditors/rc/fe/baja/OrderedMapEditor',
        'Promise',
        'underscore',
        'css!nmodule/bacnet/rc/bacnet'], function (
        baja,
        events,
        batchSaveMixin,
        subscriberMixin,
        lexs,
        BacnetAny,
        fe,
        BaseEditor,
        OrderedMapEditor,
        Promise,
        _) {

  'use strict';

  var bacnetLex = lexs[0],

      TYPE_MAP = BacnetAny.prototype.TYPE_MAP,
      TYPE = 'bacnet:BacnetAny',

      MODIFY_EVENT = events.MODIFY_EVENT,

      CHOICES = (function () {
        var map = new baja.OrderedMap();

        _.each(TYPE_MAP, function (arg) {
          var choice = arg[0],
              lex = arg[2],
              text = arg[3];

          map.put(bacnetLex.get({ key: lex, def: text }), choice);
        });

        return map;
      }());

  /**
   * Field editor for working with `bacnet:BacnetAny` instances.
   *
   * @class
   * @alias module:nmodule/bacnet/rc/fe/BacnetAnyEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   */
  var BacnetAnyEditor = function BacnetAnyEditor() {
    BaseEditor.apply(this, arguments);
    batchSaveMixin(this);
    subscriberMixin(this);
  };
  BacnetAnyEditor.prototype = Object.create(BaseEditor.prototype);
  BacnetAnyEditor.prototype.constructor = BacnetAnyEditor;

  /**
   * Set event handlers to rebuild the `value` editor depending on which choice
   * is selected.
   *
   * @param {JQuery} dom
   */
  BacnetAnyEditor.prototype.doInitialize = function (dom) {
    var that = this;

    dom.html('<div class="js-choice"/><div class="js-value"/>');

    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });

    dom.on(MODIFY_EVENT, '.js-choice', function () {
      var choice = that.$getChoiceEditor().getSelectedValue(),
          valueEd = that.$getValueEditor();

      valueEd.read()
        .then(function (value) {
          return that.$loadAny(choice, value);
        })
        .catch(baja.error);
    });

    return fe.buildFor({
      dom: dom.children('.js-choice'),
      formFactor: 'mini',
      type: OrderedMapEditor,
      value: CHOICES
    });
  };


  /**
   * @param {baja.Component} BacnetAny the `bacnet:BacnetAny` to load
   *
   * @returns {Promise}
   */
  BacnetAnyEditor.prototype.doLoad = function (bacnetAny) {
    return this.$loadAny(bacnetAny.get('choice'), bacnetAny.get('value'));
  };

  /**
   * Read a new `BacnetAny` with currently selected values.
   *
   * @returns {Promise.<baja.Component>}
   */
  BacnetAnyEditor.prototype.doRead = function () {
    return this.getChildEditors().readAll()
      .spread(function (choice, value) {
        return baja.$(TYPE, { choice: choice, value: value });
    });
  };

  /**
   * Apply the currently entered `choice` and `value` values back to the
   * `BacnetAny`. Respects the `batchSaveMixin` contract.
   *
   * @param {baja.Component} bacnetAny - a `bacnet:BacnetAny` to save
   *
   * @param {Object} params
   */
  BacnetAnyEditor.prototype.doSave = function (bacnetAny, params) {
    var that = this,
        value = that.value(),
        batch = params && params.batch,
        progressCallback = params && params.progressCallback;

    var prom = Promise.all([
          value.set({
            slot: 'choice',
            value: bacnetAny.get('choice'),
            batch: batch
          }),
          value.set({
            slot: 'value',
            value: bacnetAny.get('value'),
            batch: batch,
          })
        ]);
    if (progressCallback) { progressCallback(batchSaveMixin.COMMIT_READY); }
    return prom;
  };

  /**
   * Set all child editors enabled/disabled.
   *
   * @param {boolean} enabled
   *
   * @returns {Promise}
   */
  BacnetAnyEditor.prototype.doEnabled = function (enabled) {
    return this.getChildEditors().setAllEnabled(enabled);
  };

  /**
   * Set all child editors readonly/writable.
   *
   * @param {boolean} readonly
   *
   * @returns {Promise}
   */
  BacnetAnyEditor.prototype.doReadonly = function (readonly) {
    return this.getChildEditors().setAllReadonly(readonly);
  };

  /**
   * Destroy all child editors.
   *
   * @returns {Promise}
   */
  BacnetAnyEditor.prototype.doDestroy = function () {
    return this.getChildEditors().destroyAll();
  };

  /**
   * Update the selected choice and value values, rebuilding the `value` editor
   * in the process as the choice may have changed.
   *
   * @private
   *
   * @param {number|baja.Integer} choice
   *
   * @param {*} value
   *
   * @returns {Promise}
   */
  BacnetAnyEditor.prototype.$loadAny = function (choice, value) {
    var that = this,
        valueEd = that.$getValueEditor(),
        type =  TYPE_MAP[choice][1];

    choice = choice.valueOf();
    var getValue = baja.hasType(value, type) ? value :
          baja.importTypes([type]).then(function () { return baja.$(type); });

    return Promise.join(getValue, valueEd && valueEd.destroy())
      .spread(function (value) {
        that.$getChoiceEditor().setSelectedValue(choice, true);
        return fe.buildFor({
          value: value,
          formFactor: 'mini',
          dom: that.$getValueElement()
        });
      });
  };


  /**
   * Get the editor for the selected choice.
   *
   * @private
   *
   * @returns {module:nmodule/webEditors/rc/fe/baja/OrderedMapEditor}
   */
  BacnetAnyEditor.prototype.$getChoiceEditor = function () {
    return this.$getChoiceElement().data('widget');
  };

  /**
   * Get the editor for the `value` slot.
   *
   * @private
   *
   * @returns {*}
   */
  BacnetAnyEditor.prototype.$getValueEditor = function () {
    return this.$getValueElement().data('widget');
  };

  /**
   * @private
   *
   * @returns {JQuery} element containing the `choice` editor
   */
  BacnetAnyEditor.prototype.$getChoiceElement = function () {
    return this.jq().children('.js-choice');
  };

  /**
   * @private
   *
   * @returns {JQuery} element containing the `value` editor
   */
  BacnetAnyEditor.prototype.$getValueElement = function () {
    return this.jq().children('.js-value');
  };

  return BacnetAnyEditor;
});
