/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

define([
  'lex!baja',
  'underscore'
], function (lexs, _) {
  'use strict';

  var bajaLex = lexs[0],
    WEEKDAYS_LEX_KEYS = [
      'monday.short',
      'tuesday.short',
      'wednesday.short',
      'thursday.short',
      'friday.short',
      'saturday.short',
      'sunday.short'
    ],
    WEEKDAYS_SHORT = [
      'Mon',
      'Tue',
      'Wed',
      'Thu',
      'Fri',
      'Sat',
      'Sun'
    ];

  /**
   * Utilities for working with BacnetDates
   *
   * API Status: **Private**
   * @exports nmodule/bacnet/rc/baja/util/BacnetDateUtil
   */
  var exports = {};

  /**
   * Given an encoded bacnet date string, this function returns the localized display string
   * 
   * @param {string} bacnetDateStr the encoded date string
   * @returns {string} The localized date string given an encoded date string
   */
  exports.getDisplayBacnetDateString = function (bacnetDateStr) {
    if (bacnetDateStr) {
      var weekdayPart = bacnetDateStr.substring(bacnetDateStr.lastIndexOf('-') + 1);
      var restOfDate = bacnetDateStr.substring(0, bacnetDateStr.lastIndexOf('-'));

      var weekDayIndex = _.findIndex(WEEKDAYS_SHORT, function (weekDay) {
        return weekDay === weekdayPart;
      });
      if (weekDayIndex !== -1) {
        return restOfDate + "-" + exports.$getLocalizedShortDayOfWeek(WEEKDAYS_LEX_KEYS[weekDayIndex]);
      }
    }
    return bacnetDateStr;
  };

  /**
   * @private
   */
  exports.$getLocalizedShortDayOfWeek = function (dayOfWeekKey) {
    return bajaLex.get(dayOfWeekKey);
  };

  /**
   * Given an bacnet date display string, this function returns the string that can be deserialized
   * 
   * @param {string} bacnetDateStr the display date string
   * @returns {string} The encoded date string given a display date string
   */
  exports.getEncodedBacnetDateString = function (bacnetDateStr) {
    if (bacnetDateStr) {
      var weekdayPart = bacnetDateStr.substring(bacnetDateStr.lastIndexOf('-') + 1);
      var restOfDate = bacnetDateStr.substring(0, bacnetDateStr.lastIndexOf('-'));

      var index = _.findIndex(WEEKDAYS_LEX_KEYS, function (key) {
        return exports.$getLocalizedShortDayOfWeek(key) === weekdayPart;
      });
      if (index !== -1) {
        return restOfDate + "-" + WEEKDAYS_SHORT[index];
      }
    }
    return bacnetDateStr;
  };

  return exports;
});
