/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/baja/datatypes/BacnetTime
 */
define(['baja!',
  'bajaScript/baja/obj/objUtil'], function (baja,
                                            objUtil) {

  'use strict';

  var Simple = baja.Simple,
    cacheDecode = objUtil.cacheDecode,
    cacheEncode = objUtil.cacheEncode;

  var UNSPECIFIED = -1;
  var DEFAULT_ENCODING = "**:**:**.**";

  /**
   * Helper - Validates if the input value is in range
   * -1 and 255 are special values allowed and considered 'in' this range
   * @param {number} value - The input value
   * @param {number} max - The max value allowed
   * @returns {boolean} True if the value is in range
   */
  function inRange(value, max) {
    return !isNaN(value) && (((value >= UNSPECIFIED && value <= max)) || value === 255);
  }

  /**
   * BajaScript representation of a `bacnet:BacnetTime` value.
   *
   * @class
   * @extends {baja.Simple}
   * @alias {module:nmodule/bacnet/rc/baja/datatypes/BacnetTime}
   */
  var BacnetTime = function BacnetTime() {
    Simple.apply(this, arguments);
  };
  BacnetTime.prototype = Object.create(Simple.prototype);
  BacnetTime.prototype.constructor = BacnetTime;

  /**
   * Create a new `bacnet:BacnetTime` instance from the arguments.
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetTime}
   */
  BacnetTime.make = function (hour, minute, second, hundredth) {
    hour = parseInt(hour, 10);
    minute = parseInt(minute, 10);
    second = parseInt(second, 10);
    hundredth = parseInt(hundredth, 10);

    if (!inRange(hour, 23)) {
      throw new Error("BBacnetTime: invalid hour:" + hour);
    }

    if (!inRange(minute, 59)) {
      throw new Error("BBacnetTime: invalid minute:" + minute);
    }

    if (!inRange(second, 59)) {
      throw new Error("BBacnetTime: invalid second:" + second);
    }

    if (!inRange(hundredth, 99)) {
      throw new Error("BBacnetTime: invalid hundredth:" + hundredth);
    }

    var bt = new BacnetTime();
    bt.$hour = hour;
    bt.$minute = minute;
    bt.$second = second;
    bt.$hundredth = hundredth;

    return bt;
  };

  /**
   * @see .make
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetTime}
   */
  BacnetTime.prototype.make = function () {
    return BacnetTime.make.apply(BacnetTime, arguments);
  };

  /**
   * Decode an `BacnetTime` from a string.
   *
   * @function
   * @param {String} str
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetTime}
   */
  BacnetTime.prototype.decodeFromString = cacheDecode(function (str) {
    if (!str || str === "" || str === DEFAULT_ENCODING) {
      return BacnetTime.DEFAULT;
    }
    /**
     * The pattern to match a BacnetTime encoding against.
     * Examples of matching encodings include,
     * **:**:**.**
     * 255:**:255.255
     * 23:59:59.99
     * 01:02:04.04
     * @type {RegExp}
     */
    var encPtrn = /^(\*{2}|\d|0\d|1?\d|2[0-3]):(\*{2}|([0-5]?\d)):(\*{2}|([0-5]?\d)).(\*{2}|(\d?\d))$/;
    if (!encPtrn.test(str)) {
      throw new Error("Incorrect encoding");
    }
    var tmParts = str.split(":");
    var hr = tmParts[0] === "**" ? UNSPECIFIED : tmParts[0];
    var min = tmParts[1] === "**" ? UNSPECIFIED : tmParts[1];

    var tmpSecs = tmParts[2].split('.');
    var sec = tmpSecs[0] === "**" ? UNSPECIFIED : tmpSecs[0];
    var hdt = tmpSecs[1] === "**" ? UNSPECIFIED : tmpSecs[1];

    return BacnetTime.make(hr, min, sec, hdt);
  });

  /**
   * Encode this `BacnetTime` to a string.
   *
   * @function
   * @returns {String}
   */
  BacnetTime.prototype.encodeToString = cacheEncode(function () {
    var encStr = "";
    encStr += this.$hour === UNSPECIFIED ? "**:" : (this.$hour < 10 ? "0" + this.$hour : this.$hour) + ":";
    encStr += this.$minute === UNSPECIFIED ? "**:" : (this.$minute < 10 ? "0" + this.$minute : this.$minute) + ":";
    encStr += this.$second === UNSPECIFIED ? "**." : (this.$second < 10 ? "0" + this.$second : this.$second) + ".";
    encStr += this.$hundredth === UNSPECIFIED ? "**" : (this.$hundredth < 10 ? "0" + this.$hundredth : this.$hundredth);

    return encStr;
  });

  /**
   * Returns the hour component of time
   * @returns {Number} The hour component of time
   */
  BacnetTime.prototype.getHour = function () {
    return this.$hour;
  };
  /**
   * Returns the minute component of time
   * @returns {Number} The minute component of time
   */
  BacnetTime.prototype.getMinute = function () {
    return this.$minute;
  };
  /**
   * Returns the second component of time
   * @returns {Number} The second component of time
   */
  BacnetTime.prototype.getSecond = function () {
    return this.$second;
  };
  /**
   * Returns the hundreth of a second of time
   * @returns {Number} The hundredth component of time
   */
  BacnetTime.prototype.getHundredth = function () {
    return this.$hundredth;
  };

  BacnetTime.DEFAULT = BacnetTime.make(UNSPECIFIED, UNSPECIFIED, UNSPECIFIED, UNSPECIFIED);

  baja.registerType('bacnet:BacnetTime', function () {
    return BacnetTime;
  });
  return BacnetTime;
});

