/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/baja/datatypes/BacnetObjectIdentifier
 */
define([
  'baja!',
  'bajaScript/baja/obj/objUtil',
  'nmodule/bacnet/rc/baja/enums/BacnetObjectType',
  'lex!bacnet',
  'baja!bacnet:BacnetObjectType'], function
  (baja,
   objUtil,
   BacnetObjectType,
   lexs) {

  'use strict';

  /**
   * Note: This number is allowed but is reserved. Max is actually 4194302
   */
  var UNCONFIGURED_INSTANCE_NUMBER = 4194303;

  var bacLex = lexs[0];

  var Simple = baja.Simple,
    cacheDecode = objUtil.cacheDecode,
    cacheEncode = objUtil.cacheEncode;

  /**
   * BajaScript representation of a `bacnet:BacnetObjectIdentifier` value.
   *
   * @class
   * @extends {baja.Simple}
   * @alias {module:nmodule/bacnet/rc/baja/datatypes/BacnetObjectIdentifier}
   */
  var BacnetObjectIdentifier = function () {
    Simple.apply(this, arguments);
  };
  BacnetObjectIdentifier.prototype = Object.create(Simple.prototype);
  BacnetObjectIdentifier.prototype.constructor = BacnetObjectIdentifier;

  /**
   * Create a new `bacnet:BacnetObjectIdentifier` instance from the arguments.
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetObjectIdentifier}
   */
  BacnetObjectIdentifier.make = function (objectType, instanceNumber) {
    var objType = parseInt(objectType, 10),
        objInst = parseInt(instanceNumber, 10);

    if (isNaN(objType) || objType < 0) {
      throw new Error(bacLex.get("BacnetObjectIdentifier.invalid.objectType"));
    }

    if (isNaN(objInst) || objInst > UNCONFIGURED_INSTANCE_NUMBER) {
      throw new Error(bacLex.get("bacnetObjectIdentifier.invalidInstanceNumber"));
    }

    var bObj = new BacnetObjectIdentifier();
    bObj.$objType = objType;
    bObj.$inst = objInst;
    return bObj;
  };

  /**
   * @see BacnetObjectIdentifier.make
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetObjectIdentifier}
   */
  BacnetObjectIdentifier.prototype.make = function () {
    return BacnetObjectIdentifier.make.apply(BacnetObjectIdentifier, arguments);
  };

  /**
   * Decode a `bacnet:BacnetObjectIdentifier` from a string.
   *
   * @function
   * @param {String} str
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetObjectIdentifier}
   */
  BacnetObjectIdentifier.prototype.decodeFromString = cacheDecode(function (str) {
    if (!str) {
      return BacnetObjectIdentifier.DEFAULT;
    }
    var idParts = str.split(':');

    return BacnetObjectIdentifier.make(BacnetObjectType.idForTag(idParts[0]), idParts[1]);
  });

  /**
   * Encode this `BacnetObjectIdentifier` to a string.
   *
   * @function
   * @returns {String}
   */
  BacnetObjectIdentifier.prototype.encodeToString = cacheEncode(function () {
    return BacnetObjectType.tagForId(this.$objType) + ":" + this.$inst;
  });

  /**
   * Get the object type of this bacnet object
   * @returns {Number} The object type id
   */
  BacnetObjectIdentifier.prototype.getObjectType = function () {
    return this.$objType;
  };
  /**
   * Get the instance number of this bacnet object
   * @returns {Number} The instance number of this bacnet object
   */
  BacnetObjectIdentifier.prototype.getInstanceNumber = function () {
    return this.$inst;
  };

  BacnetObjectIdentifier.DEFAULT = BacnetObjectIdentifier.make(0, -1);

  baja.registerType('bacnet:BacnetObjectIdentifier', function () {
    return BacnetObjectIdentifier;
  });

  return BacnetObjectIdentifier;
});
