/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/baja/datatypes/BacnetAddress
 */
define([
  'baja!',
  'bajaScript/baja/obj/objUtil',
  'underscore'], function
  (baja,
   objUtil,
   _) {

  'use strict';

  var Struct         = baja.Struct,
      ETHERNET_REGEX = /^([0-9A-Fa-f]{1,2}[:-]){5}([0-9A-Fa-f]{1,2})$/,
      MSTP_REGEX     = /^(-)?\d+$/,
      IP_REGEX       = new RegExp([
        '^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}',
        '(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)(:0x[0-9A-Fa-f]{1,4})?$'
      ].join('')),

      UNKNOWN = 0,
      ETHERNET = 1,
      IP = 2,
      MSTP = 3;

  function byteToHexString(byte) {
    byte = (byte || 0) & 0xff;
    var s = byte.toString(16);
    if (s.length === 1) {
      s = '0' + s;
    }
    return s.toUpperCase();
  }

  function bytesToHexString(bytes, seperator) {
    bytes = bytes || [];
    return _.map(bytes, byteToHexString).join(seperator);
  }

  function isNegativeByte(byte) {
    return byte < 0;
  }

  /**
   * BajaScript representation of an `bacnet:BacnetAddress` value.
   *
   * @class
   * @extends {baja.Simple}
   * @alias {module:nmodule/bacnet/rc/baja/datatypes/BacnetAddress}
   */
  var BacnetAddress = function BacnetAddress() {
    Struct.apply(this, arguments);
  };
  BacnetAddress.prototype = Object.create(Struct.prototype);
  BacnetAddress.prototype.constructor = BacnetAddress;

  /**
   * Encodes a byte array to a MAC Address string formatted according to the 'type'
   * parameter
   * @param {Number} type MAC Address integer type
   * @param {Array} bytes Byte Array to be encoded
   * @returns {String} Encoded MAC Address string
   */
  BacnetAddress.bytesToString = function (type, bytes) {
    if (!bytes || !bytes.length) {
      return 'null';
    }

    switch (type) {
      case ETHERNET:
        if (bytes.length !== 6) {
          throw  new Error("Invalid Ethernet MAC address!");
        }
        return bytesToHexString(bytes, ':');

      case IP:
        if (bytes.length < 4 || bytes.length > 4 && bytes.length !== 6) {
          throw new Error("Invalid IP MAC address!");
        }

        var str = '',
            port;

        if (bytes.length >= 4) {
          str = _.map(bytes.slice(0, 4), function (byte) {
            return byte & 0xff;
          }).join('.');
        }

        if (bytes.length === 6) {
          port = (bytes[4] & 0xff) << 8 | (bytes[5] & 0xff);
          str = str + ':0x' + port.toString(16).toUpperCase();
        }

        return str;

      default:
        if (bytes.length === 1) {
          return (bytes[0] & 0xff).toString();
        }

        else {
          return bytesToHexString(bytes, ' ');
        }
    }
  };

  /**
   * Decodes a MAC Address string to a byte array
   * @param {Number} type MAC Address integer type
   * @param {Number} len Number of tokens in byte string
   * @param {String} str MAC Address string encoding
   * @returns {Array} Array of bytes
   */
  BacnetAddress.stringToBytes = function (type, len, str) {
    if (!str || !str.length || str === 'null') {
      return null;
    }

    var tokens,
        bytes;
    switch (type) {
      case ETHERNET:

        if (!str.match(ETHERNET_REGEX)) {
          throw new Error("Invalid Ethernet MAC Address!");
        }

        tokens = str.split(/[:\s]/);
        if (tokens.length !== 6) {
          throw new Error("Invalid Ethernet MAC Address!");
        }
        bytes = _.map(tokens, function (token) {
          return parseInt(token, 16);
        });

        if (_.some(bytes, isNegativeByte)) {
          throw new Error("Invalid Ethernet MAC Address!");
        }

        return bytes;

      case IP:

        if (!str.match(IP_REGEX)) {
          throw new Error("Invalid BACnet/IP MAC Address!");
        }

        tokens = str.split(/[.:\s]/);

        if (tokens.length !== len) {
          throw new Error("Invalid BACnet/IP MAC Address!");
        }

        if (tokens.length >= 4) {
          bytes = _.map(tokens.slice(0, 4), function (token) {
            return parseInt(token, 10);
          });

          if (_.some(bytes, isNegativeByte)) {
            throw new Error("Invalid BACnet/IP MAC Address!");
          }

          switch (tokens.length) {
            case 4:
              return bytes;
            case 5:
              var port = parseInt(tokens[4], 16);
              if (port < 0) {
                throw new Error("Invalid BACnet/IP MAC Address!");
              }
              return bytes.concat([(port >> 8) & 0xff, port & 0xff]);
            default:
              throw new Error("Invalid length for conversion of BACnet/IP MAC address!");
          }
        }
        break;

      default:
        tokens = str.split(/[:\s]/);
        if (tokens.length === 1) {
          if(!str.match(MSTP_REGEX)){
            throw new Error("Invalid BACnet/IP MSTP Address!");
          }
          return [parseInt(tokens[0], 10) & 0xff];
        }

        else {
          return _.map(tokens, function (token) {
            return parseInt(token, 16);
          });
        }
    }
  };

  /**
   * Translates a given formatted MAC Address String to the target format
   * specified by the 'type' argument
   * @param {Number} type Target integer type to translate MAC Address to
   * @param {String} macStr MAC Address string in a given format
   * @returns {String} Translated MAC Address String
   */
  BacnetAddress.convertMacAddress = function (type, macStr) {
    if (!macStr || macStr === 'null') {
      return macStr;
    }

    var tokens;

    switch (type) {
      case ETHERNET:
        if (macStr.indexOf('.') === -1) {
          return _.map(macStr.split(/[:\s]/), function (token) {
            return token.toUpperCase();
          }).join(':');
        }

        tokens = macStr.split(/[.:\s]/);
        if (tokens.length < 5) {
          return macStr;
        }

        return _.map(tokens, function (token, index) {
          if (index === 4) {
            var port = parseInt(token, 16);
            return bytesToHexString([port >> 8, port & 0xff], ':');
          }
          return byteToHexString(parseInt(token, 10));
        }).join(':');


      case IP:
        if (macStr.indexOf('.') === -1) {
          tokens = macStr.split(/[:\s]/);
          if (tokens.length < 6) {
            return macStr;
          }

          var ipStr = _.map(tokens.slice(0, 4), function (token) {
            return parseInt(token, 16);
          }).join('.');

          ipStr = ipStr + ":0x";
          var port = (parseInt(tokens[4], 16) & 0xff) << 8;
          port = port | (parseInt(tokens[5], 16) & 0xff);
          ipStr = ipStr + port.toString(16).toUpperCase();
          return ipStr;
        }
        return macStr;

      default:
        if (macStr.indexOf('.') === -1) {
          tokens = macStr.split(/[:\s]/);
          return _.map(tokens, function (token) {
            return token.toUpperCase();
          }).join(' ');
        }

        tokens = macStr.split(/[.:]/);
        if (tokens.length < 5) {
          return macStr;
        }

        return _.map(tokens, function (token, index) {
          if (index === 4) {
            var port = parseInt(token, 16);
            return bytesToHexString([(port >> 8) & 0xff, port & 0xff], ' ').toUpperCase();
          }
          return byteToHexString(parseInt(token)).toUpperCase();
        }).join(' ');
    }
  };

  /**
   * Return the String value of the address formatted based on addressType.
   *
   * @returns {String}
   */
  BacnetAddress.prototype.toString = function () {
    var that = this,
        str = that.getNetworkNumber() + ':',
        bytes = that.getMacAddress().getBytes();

    function toByte (byte){
      return byte & 0xFF;
    }

    if(bytes.length === 0) {
      str += 'null';
    } else {
      switch(that.getAddressType()) {
        case IP:
          str += _.map(bytes.slice(0,4), function (byte) {
              return toByte(byte);
            }).join('.') + ':' +
            (((toByte(bytes[4])<<8))|toByte(bytes[5]));
          break;
        case MSTP:
        case UNKNOWN:
        case ETHERNET:
          str += bytesToHexString(bytes, ' ').toLowerCase();
          break;
      }
    }
    return str;
  };

  BacnetAddress.MAC_TYPE_UNKNOWN = UNKNOWN;
  BacnetAddress.MAC_TYPE_ETHERNET = ETHERNET;
  BacnetAddress.MAC_TYPE_IP = IP;
  BacnetAddress.MAC_TYPE_MSTP = MSTP;

  baja.registerType('bacnet:BacnetAddress', function () {
    return BacnetAddress;
  });

  return BacnetAddress;
});

