/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/* eslint-env browser */

/**
 * The bajaux Container for a Workbench environment.
 * This Container works in conjunction with custom Java callbacks
 * so the widget can load as appropriate in a Workbench Java environment.
 * This JavaScript is only intended to run in the Workbench Web Browser.
 *
 * @module bajaux/container/wb/wbContainer
 * @private
 */
define(["jquery", "Promise", "underscore", "baja!", "bajaux/events", "bajaux/commands/Command", "bajaux/container/wb/Clipboard", "bajaux/container/util", "bajaux/util/CommandButtonGroup", "dialogs", "lex!bajaux", "hbs!bajaux/container/error"], function ($, Promise, _, baja, events, Command, Clipboard, util, CommandButtonGroup, dialogs, lex, errorTemplate) {
  "use strict";

  lex = lex[0];

  function defer() {
    var res,
        rej,
        // eslint-disable-next-line promise/avoid-new
    promise = new Promise(function (resolve, reject) {
      res = resolve;
      rej = reject;
    });
    return {
      resolve: res,
      reject: rej,
      promise: promise
    };
  }

  var exports = {},
      initDf,
      loadDf,
      containerJq,
      jq,
      errorJq,
      bodyJq,
      setValueParam = "ext",
      setMetadataParam = "ext",
      detachDom = false,
      // Switched off by default as it's causing too many problems.
  containerOptions,
      updateProperties,
      UNRECOVERABLE_FAIL_EVENTS = [events.INITIALIZE_FAIL_EVENT, events.LOAD_FAIL_EVENT, events.DESTROY_FAIL_EVENT],
      RUNTIME_FAIL_EVENTS = [events.ENABLE_FAIL_EVENT, events.DISABLE_FAIL_EVENT, events.SAVE_FAIL_EVENT, events.READONLY_FAIL_EVENT, events.WRITABLE_FAIL_EVENT //        events.command.FAIL_EVENT //SAVE_FAIL_EVENT and command FAIL_EVENT double up.
  ],
      isSavePending = false;
  baja.comm.setCommFailCallback(function commFail(err) {
    // TODO: once Workbench dialogs are implemented, we should make a pop up here
    // for this error (just like Hx).
    baja.error("Comms fail: " + err);
  });
  /**
   * Used to convert an ORD to a hyperlink. In the case of Workbench, no conversion is required
   * including replacement of backups ("../").  For details regarding backups replacement see
   * NCCB-18551.
   *
   * @inner
   *
   * @param  {String} ordStr The ORD String.
   * @returns {Promise} The promise that resolves with the
   * converted ORD.
   */

  function toHyperlink(ordStr) {
    return Promise.resolve(ordStr);
  }
  /**
   * Translate a drag operation into Workbench world.
   *
   * @inner
   *
   * @param {String|Object} json a JSON encoded string, or an object to
   * be encoded to JSON. The encoded object should have a String `mime`
   * property, and an Array `data` property.
   * @param {JQuery.Event} ev the jQuery dragstart event
   * @returns {boolean} false to cancel?
   */


  function startDrag(json, ev) {
    var wbutil = window.niagara.wb.util,
        modifiers = 0;

    if (ev.shiftKey) {
      modifiers |= 1;
    }

    if (ev.ctrlKey) {
      modifiers |= 2;
    }

    if (ev.altKey) {
      modifiers |= 8;
    }

    if (typeof json !== "string") {
      json = $.isArray(json) ? json : [json];
      json = JSON.stringify(json);
    } // This corresponds to BWebWidget$WebWidgetUtil#startDrag


    wbutil.startDrag(JSON.stringify({
      drag: json,
      x: ev.pageX || 0,
      y: ev.pageY || 0,
      modifiers: modifiers
    }));
    return false;
  }
  /**
   * Close any open dialogs boxes.
   */


  function closeAllDialogs() {
    dialogs.each(function (i, dlg) {
      dlg.close();
    });
  }
  /**
   * Called when an error occurs in bajaux.
   *
   * @private
   *
   * @param  widget The widget instance the error occurred on.
   * @param  {String} type The type of error that occurred.
   * @param  err The error information.
   */


  function error(widget, type, err) {
    baja.error(err);
    jq.hide();
    var params = {
      title: lex.get("error.title"),
      message: lex.get("error.message"),
      details: lex.get("error.details"),
      type: type,
      moduleName: widget ? widget.$moduleName : "bajaux",
      keyName: widget ? widget.$keyName : "widget",
      stack: err + (err && err.stack ? '\n' + err.stack : '')
    }; // Remove any existing error messages

    errorJq.empty();
    errorJq.html(errorTemplate(params));
    errorJq.show();
    closeAllDialogs();
    var wbWidget = window.wbWidget;

    if (wbWidget) {
      // Remove the widget
      delete window.wbWidget;

      if (wbWidget.isInitialized()) {
        wbWidget.destroy();
      } // This should stop the commands from being shown.


      window.niagara.wb.util.buildCommands(null);
    }

    window.niagara.wb.util.error(err instanceof Error ? err.message : err);
  }
  /**
   * Handles runtime errors that are triggered. If save is pending, the error will
   * not display until after the save if complete.
   * @param {jQuery.Event} e
   * @param {module:bajaux/Widget} sourceWidget
   * @param {Error} err
   */


  function handleRuntimeError(e, sourceWidget, err) {
    var widget = window.wbWidget;
    var isBaseWidget = widget && widget === sourceWidget;

    if (isBaseWidget) {
      baja.error(err);
      var error = err instanceof Error ? err.message : err;

      if (isSavePending) {
        showNativeErrorDialog(error);
      } else {
        dialogs.showOk({
          title: lex.get('error.title'),
          content: '<span class="js-dialog-wrapTextContent">' + _.escape(error) + '</span>'
        });
      }
    }
  }
  /**
   * Shows a native error dialog. Requires widget to be in the save process for
   * dialog to display.
   * @param {string} msg
   */


  function showNativeErrorDialog(msg) {
    var wbutil = window.niagara.wb.util;
    wbutil.saveFail(msg);
  }

  function reattachContainer() {
    if (!containerJq.parent().length) {
      bodyJq.append(containerJq);

      if (window.wbWidget && window.wbWidget.isInitialized()) {
        window.wbWidget.layout();
      }
    }
  }

  function updateProperty(properties, propName, updatedProp) {
    var existingProp = properties.get(propName),
        valueDefined = updatedProp.value !== undefined,
        metadataDefined = updatedProp.metadata !== undefined;

    if (existingProp && !existingProp.userModified) {
      if (valueDefined || metadataDefined) {
        if (valueDefined) {
          properties.setValue(propName, updatedProp.value, setValueParam);
        }

        if (metadataDefined) {
          properties.setMetadata(propName, updatedProp.metadata, setMetadataParam);
        }
      } else {
        properties.setValue(propName, updatedProp, setValueParam);
      }
    } else if (!existingProp && propName === "enabled") {
      properties.add({
        name: propName,
        value: valueDefined ? updatedProp.value : updatedProp
      });

      if (metadataDefined) {
        properties.setMetadata(propName, updatedProp.metadata, setMetadataParam);
      }
    }
  }

  $(window).resize(function () {
    if (window.wbWidget && window.wbWidget.isInitialized()) {
      window.wbWidget.layout();
    }
  });
  /**
   * Sets the options for the life time of the container. This is called once
   * when the container is first created.
   *
   * @param {Object} options An object that contains the options for
   * the container.
   */

  exports.initializeContainer = function initializeContainer(options) {
    containerOptions = options;
    containerJq = $("#bajaux-container");
    jq = $("#bajaux-widget");
    errorJq = $("#bajaux-error");
    bodyJq = $("body");
    bodyJq.css('margin', '0px');
    /* NCCB-20798 */
  };
  /**
   * Update a Widget's properties.
   *
   * @param {String} propsStr A JSON encoded string that contains the properties we're going to update.
   */


  exports.updateProperties = updateProperties = function updateProperties(propsStr) {
    var widget = window.wbWidget;

    if (propsStr && widget) {
      var existingProps = widget.properties(),
          updatedProps = JSON.parse(propsStr);
      Object.keys(updatedProps).forEach(function (propName) {
        updateProperty(existingProps, propName, updatedProps[propName]);
      });
    }
  };
  /**
   * Handles the initialize lifecycle of a widget.
   *
   * @private
   *
   * @param widget The widget to be initialized.
   * @param  {Object} parameters Parameters to load as widget properties.
   * @returns {Promise} Returns a promise that will be resolved once
   * the widget has been initialized.
   */


  exports.initialize = function initialize(widget, parameters) {
    var wbutil = window.niagara.wb.util,
        env = window.niagara.env,
        failureEvents = UNRECOVERABLE_FAIL_EVENTS.join(" "),
        properties = widget.properties(); // Remove the container from the DOM while loading.

    if (detachDom) {
      containerJq.remove();
    }

    initDf = defer();

    if (env) {
      jq.toggleClass("bajaux-design-time", !!env.designTime);
      env.toHyperlink = toHyperlink;
      env.startDrag = startDrag;
    } // If a widget is currently loaded then destroy it.


    function cleanup() {
      closeAllDialogs();
      errorJq.empty().hide();
      jq.show();
      var wbWidget = window.wbWidget;

      if (wbWidget) {
        delete window.wbWidget;

        if (wbWidget.isInitialized()) {
          wbutil.destroying();
          util.preDestroy(wbWidget);
          return wbWidget.destroy().then(function () {
            wbutil.destroyed();
          });
        }
      }
    }

    function armFailureHandlers() {
      // Listen for failure events and only handle this once.
      jq.one(failureEvents, function (ev, v, err) {
        error(widget, ev.type, err);
      });
      jq.on(RUNTIME_FAIL_EVENTS.join(' '), handleRuntimeError);
    }

    function errorHandler(err) {
      reattachContainer();
      initDf.reject(err);

      if (window.wbWidget) {
        delete window.wbWidget;
      }

      throw err;
    }

    function propsFromParameters() {
      if (parameters) {
        $.each(parameters, function (name, value) {
          value = util.decodeValueFromParameter(properties, name, value);

          if (value !== null) {
            properties.setValue(name, value, setValueParam);
          }
        });
      }
    }
    /**
     * Configure the enabled state based off the initial value of the enabled property.
     * @return {Promise}
     */


    function configureEnabled() {
      if (properties.getValue("enabled") === false && widget.isEnabled()) {
        return widget.setEnabled(false);
      }

      return Promise.resolve();
    } // Resolve and initialize simultaneously as they're
    // independent of each other.


    function init() {
      // Cache the widget instance so it can accessed globally
      window.wbWidget = widget;
      updateProperties(wbutil.getProperties());
      return configureEnabled().then(function () {
        wbutil.initializing();
        return widget.initialize(jq);
      });
    }

    function initCommands() {
      if (wbutil.isCommandBarVisible()) {
        return util.initCommandBar(widget, containerJq);
      }
    }

    function encodeCommand(cmd) {
      var data = {},
          promises = [];

      if (cmd.isCommand()) {
        data.id = cmd.getId();
        data.toggle = cmd.isToggleCommand(); //here we are switching back to module prefix for Workbench commands

        data.icon = cmd.getIcon().replace("https://workbench/module/", "module://");
        data.icon = data.icon.replace("/module/", "module://");
        data.enabled = cmd.isEnabled();
        data.menu = cmd.hasFlags(Command.flags.MENU_BAR);
        data.toolbar = cmd.hasFlags(Command.flags.TOOL_BAR);
        data.selected = cmd.isToggleCommand() && cmd.isSelected();
        data.command = true;
        promises.push(cmd.toDescription().then(function (description) {
          data.description = description;
        }));
      } else {
        data.command = false;
        data.kids = [];
      }

      promises.push(cmd.toDisplayName().then(function (displayName) {
        data.displayName = displayName;
      }));
      return Promise.all(promises).then(function () {
        return data;
      });
    }

    function encodeCommandTree(cmd) {
      return encodeCommand(cmd).then(function (data) {
        return cmd.isCommand() ? data : Promise.all(_.map(cmd.getChildren(), function (c) {
          return encodeCommandTree(c);
        })).then(function (kids) {
          data.kids = kids;
          return data;
        });
      });
    }
    /**
     * @returns {Promise}
     */


    function buildCommands() {
      return widget.getCommandGroup().loading().then(function () {
        return encodeCommandTree(widget.getCommandGroup());
      }).then(function (data) {
        wbutil.buildCommands(JSON.stringify(data));
      });
    }

    function syncProperties(data) {
      updateProperties(wbutil.syncProperties(JSON.stringify(data)));
    }

    function register() {
      // Attach handlers to listen to Commands
      jq.on(events.command.CHANGE_EVENT, function (event, cmd) {
        encodeCommand(cmd).then(function (data) {
          wbutil.updateCommand(JSON.stringify(data));
        })["catch"](baja.error);
      }); // Attach handlers to listen to CommandGroups

      jq.on(events.command.GROUP_CHANGE_EVENT, buildCommands); // When the widget is modified, update Workbench.

      jq.on(events.MODIFY_EVENT, function () {
        wbutil.setModified();
      }); // When the widget is unmodified, update Workbench.

      jq.on(events.UNMODIFY_EVENT, function () {
        wbutil.clearModified();
      }); // When Properties are added and removed, update Workbench.

      jq.on(events.PROPERTY_ADDED + " " + events.PROPERTY_REMOVED, function (ev, w) {
        if (w !== widget) {
          return;
        }

        syncProperties(properties.get());
      }); // When a Property is changed, update Workbench.

      jq.on(events.PROPERTY_CHANGED, function (ev, w, name, value, params) {
        if (params !== setValueParam && w === widget) {
          wbutil.propertyChanged(name, value);
        }
      }); // When a Property's metadata is changed, update Workbench.

      jq.on(events.METADATA_CHANGED, function (ev, w, name, metadata, params) {
        if (params !== setMetadataParam && w === widget) {
          wbutil.metadataChanged(name, metadata);
        }
      }); // When the Command Group changes, reinitialize the Command Bar.

      jq.on(events.command.GROUP_CHANGE_EVENT, initCommands); // When a save happens, call the Workbench save command on everything.

      jq.on(events.SAVE_EVENT, function () {
        wbutil.save();
      }); // Now the widget has loaded, register it with Workbench.

      wbutil.initialized();
      buildCommands(); // Synchronize the Properties.

      syncProperties(properties.get());

      if (containerOptions.attachAfterInit) {
        reattachContainer();
      } // Resolve the promise


      initDf.resolve();
    }

    return Promise.resolve(armFailureHandlers()).then(cleanup).then(propsFromParameters).then(init).then(initCommands).then(register)["catch"](errorHandler);
  };

  var currentLoad;
  /**
   * Handles the load lifecycle for a widget. This will
   * resolve an ORD and load that into the widget.
   *
   * @private
   *
   * @param  {String} ord The ORD to resolve and load into the widget.
   * @returns {Promise} Returns a promise that will be resolved
   * once the widget has loaded.
   */

  exports.load = function load(ord) {
    var resolveProm,
        widget,
        wbutil = window.niagara.wb.util;

    if (!initDf) {
      return Promise.reject(new Error("Not initialized"));
    }

    loadDf = defer();
    return currentLoad = Promise.resolve(currentLoad).then(function () {
      if (window.wbWidget) {
        resolveProm = window.wbWidget.resolve(ord);
      }

      return initDf.promise.then(function () {
        widget = window.wbWidget;

        if (resolveProm) {
          return resolveProm;
        } else {
          return widget.resolve(ord);
        }
      }).then(function (value) {
        return widget.load(value);
      }).then(function () {
        reattachContainer();
        wbutil.loaded();
        loadDf.resolve();
        return widget.layout();
      })["catch"](function (err) {
        reattachContainer();
        loadDf.reject(err);
        throw err;
      });
    });
  };
  /**
   * Handles the save lifecycle of a widget.
   *
   * @private
   *
   * @returns {Promise} Returns a promise that will be
   * resolved once the widget has been saved.
   */


  exports.save = function save() {
    var wbutil = window.niagara.wb.util,
        widget = window.wbWidget; // eslint-disable-next-line promise/avoid-new

    return new Promise(function (resolve, reject) {
      function ok() {
        wbutil.saveOk();
        resolve();
      }

      function fail(err) {
        wbutil.saveFail(err ? err.toString() : "Error saving value!");
        reject(err);
      }

      function doSave() {
        try {
          if (widget && widget.isModified()) {
            // Save the widget
            isSavePending = true;
            widget.save().then(ok, fail)["finally"](function () {
              isSavePending = false;
            });
          } else {
            ok();
          }
        } catch (e) {
          fail(e);
        }
      }

      if (loadDf || initDf) {
        (loadDf || initDf).promise.then(doSave, fail);
      } else {
        doSave();
      }
    });
  };
  /**
   * Handles the destroy lifecycle of a widget.
   *
   * @private
   *
   * @returns {Promise} Returns a promise that's resolved
   * once the widget has been fully destroyed and deactivated.
   */


  exports.deactivate = function deactivate() {
    var widget = window.wbWidget,
        wbutil = window.niagara.wb.util; // eslint-disable-next-line promise/avoid-new

    return new Promise(function (resolve, reject) {
      function ok() {
        reattachContainer();
        wbutil.destroyed();
        wbutil.deactivated();
        initDf = null;
        loadDf = null;
        resolve();
      }

      function fail(err) {
        reattachContainer();
        wbutil.deactivated();
        error(widget, "deactivate", err);
        baja.error(err);
        initDf = null;
        loadDf = null;
        reject(err);
      }

      function doDeactivate() {
        try {
          closeAllDialogs();

          if (widget && widget.isInitialized()) {
            if (detachDom) {
              containerJq.remove();
            }

            wbutil.destroying();
            util.preDestroy(widget);
            widget.destroy().then(ok, fail);
          } else {
            ok();
          }
        } catch (err) {
          fail(err);
        }
      }

      currentLoad = null;

      if (loadDf || initDf) {
        (loadDf || initDf).promise.then(doDeactivate, fail);
      } else {
        doDeactivate();
      }
    });
  };
  /**
   * Called from Workbench when a drag over operations happens.
   *
   * @private
   * @function
   */


  exports.dragover = Clipboard.dragover;
  /**
   * Called from Workbench when a drop operation happens.
   *
   * @private
   * @function
   */

  exports.drop = Clipboard.drop; // Export this as global wbContainer so Workbench can
  // access this through the Web Engine.

  window.wbContainer = exports;
  return exports;
});
