/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * API Status: **Private**
 * @module nmodule/driver/rc/fe/BaseTabularConversionEditor
 */
define(['nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/config/CompositeBuilder', 'nmodule/webEditors/rc/fe/baja/DisplayOnlyEditor', 'bajaux/Widget', 'bajaux/commands/Command', 'bajaux/util/CommandButtonGroup', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'jquery', 'Promise', 'baja!', 'lex!driver', 'nmodule/driver/rc/fe/XYPointEditor', 'nmodule/driver/rc/baja/XYPoint', 'underscore', 'hbs!nmodule/driver/rc/fe/templates/BaseTabularConversionEditor', 'nmodule/webEditors/rc/util/htmlUtils', 'bajaux/events', 'css!nmodule/driver/rc/driver'], function (CompositeEditor, CompositeBuilder, DisplayOnlyEditor, Widget, Command, CommandButtonGroup, fe, feDialogs, $, Promise, baja, lexs, XYPointEditor, XYPoint, _, templateEditor, htmlUtils) {
  'use strict';

  var escapeHtml = htmlUtils.escapeHtml; /////////////////////////////////////////////////////////////////
  //Support Functions
  //////////////////////////////////////////////////////////////////

  function readPoints(builder, keys, ed) {
    return Promise.join(ed.read(), Promise.all(_.map(keys, function (key) {
      return builder.getEditorFor(key).read();
    }))).spread(function (tc, pnts) {
      return ed.value().make(tc.getDescription(), pnts);
    });
  } ////////////////////////////////////////////////////////////////
  // Commands
  ////////////////////////////////////////////////////////////////

  /**
   * Command to add a new point spec with x and y values.
   *
   * @inner
   * @class
   * @extends module:bajaux/commands/Command
   * @param {module:nmodule/driver/rc/fe/BaseTabularConversionEditor} ed
   */


  var AddCommand = function AddCommand(ed) {
    Command.call(this, {
      module: 'driver',
      lex: 'commands.addPoint',
      func: function func() {
        return feDialogs.showFor({
          type: XYPointEditor,
          value: new XYPoint({
            x: 0.0,
            y: 0.0
          })
        }).then(function (xyPoint) {
          if (!xyPoint) {
            return;
          }

          return ed.read().then(function (tc) {
            return ed.load(ed.value().make(tc.getDescription(), tc.getPoints().concat([xyPoint])));
          });
        });
      }
    });
  };

  AddCommand.prototype = Object.create(Command.prototype);
  AddCommand.prototype.constructor = AddCommand;
  /**
   * Command to delete all added points.
   * A default configuration comprising two points "s|0,0;100,100"
   * shall be retained.
   *
   * @inner
   * @class
   * @extends module:bajaux/commands/Command
   * @param {module:nmodule/driver/rc/fe/BaseTabularConversionEditor} ed
   */

  var DeleteAllCommand = function DeleteAllCommand(ed) {
    Command.call(this, {
      module: 'driver',
      lex: 'commands.deleteAllPoint',
      func: function func() {
        return ed.load(baja.$(ed.value().getType()));
      }
    });
  };

  DeleteAllCommand.prototype = Object.create(Command.prototype);
  DeleteAllCommand.prototype.constructor = DeleteAllCommand;
  /**
   * Command to resort the points in ascending order of the source (x)
   *
   * @inner
   * @class
   * @extends module:bajaux/commands/Command
   * @param {module:nmodule/driver/rc/fe/BaseTabularConversionEditor} ed
   */

  var ResortCommand = function ResortCommand(ed) {
    Command.call(this, {
      module: 'driver',
      lex: 'commands.resortPoint',
      func: function func() {
        return ed.read().then(function (tc) {
          var pts = tc.getPoints();
          pts.sort(function (a, b) {
            return a.xValue() - b.xValue();
          });
          return ed.load(ed.value().make(tc.getDescription(), pts));
        });
      }
    });
  };

  ResortCommand.prototype = Object.create(Command.prototype);
  ResortCommand.prototype.constructor = ResortCommand;
  /**
   * Command to remove a selected point.
   *
   * @inner
   * @class
   * @extends module:bajaux/commands/Command
   * @param {module:nmodule/driver/rc/fe/BaseTabularConversionEditor} ed
   */

  var RemoveCommand = function RemoveCommand(ed) {
    Command.call(this, {
      module: 'driver',
      lex: 'commands.removePoint',
      func: function func() {
        var focusedEd = ed.$focusedEditor,
            builder = ed.getBuilder();

        if (focusedEd) {
          return Promise.resolve(builder.getKeys()).then(function (keys) {
            if (keys.length > 2) {
              var selectedKey = focusedEd.jq().data('key'),
                  keysToKeep = _.filter(keys, function (key) {
                return key !== selectedKey;
              });

              return readPoints(builder, keysToKeep, ed).then(function (tc) {
                return ed.doLoad(tc);
              });
            }
          });
        }
      }
    });
  };

  RemoveCommand.prototype = Object.create(Command.prototype);
  RemoveCommand.prototype.constructor = RemoveCommand;
  /**
   * An abstract editor class specifically for all tabular conversion types. These types are,
   * * kitIo:TabularConversion
   * * ndio:TabularThermistorConversion
   * * nrio:NrioTabularThermistorConversion
   *
   * Subclasses will have to implement
   * * getInitDomProps
   * * doRead
   * * getValueClass
   *
   * Note: This editor is not intended to be extended by any other type other than the types
   * mentioned here.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   * @alias module:nmodule/driver/rc/BaseTabularConversionEditor
   */

  var BaseTabularConversionEditor = function BaseTabularConversionEditor() {
    var that = this;
    CompositeEditor.apply(that, arguments);

    if (that.getFormFactor() === Widget.formfactor.mini) {
      DisplayOnlyEditor.$mixin(this);
    }

    that.getCommandGroup().add(new AddCommand(that), new DeleteAllCommand(that), new ResortCommand(that), new RemoveCommand(that));
  }; //extend and set up prototype chain


  BaseTabularConversionEditor.prototype = Object.create(CompositeEditor.prototype);
  BaseTabularConversionEditor.prototype.constructor = BaseTabularConversionEditor;
  /**
   * @overrides module:nmodule/webEditors/rc/fe/CompositeEditor
   * @returns {*}
   */

  BaseTabularConversionEditor.prototype.makeBuilder = function () {
    var that = this;
    var builder = new CompositeBuilder();

    builder.getDomFor = function (i) {
      return $('<div class="point-row" data-key="' + i + '"></div>').appendTo(that.$getContentElement());
    };

    builder.getKeys = function () {
      return _.map(this.getDataSource().getPoints(), function (xypoint, i) {
        return String(i);
      });
    };

    builder.getValueFor = function (i) {
      return this.getDataSource().getPoints()[i] || null;
    };

    builder.getConfigFor = function (i) {
      return {
        type: XYPointEditor,
        formFactor: 'mini'
      };
    };

    return builder;
  };
  /**
   * When in mini/display-only mode, just display the description of the conversion
   * @param {module:nmodule/driver/rc/baja/TabularConversion} tblConversion
   * @returns {string}
   */


  BaseTabularConversionEditor.prototype.valueToString = function (tblConversion) {
    return tblConversion ? "[" + escapeHtml(tblConversion.getDescription()) + "]" : "";
  };
  /**
   * Pass the label texts needed for the editor in the form,
   *
   * @example
   *   labels: {
   *     desc: 'MyDesc',
   *     source: 'MySource',
   *     result: 'MyResult'
   *   }
   * @return {Object} Object of the form { labels: {desc:*, source:*, result:*} }
   */


  BaseTabularConversionEditor.prototype.getInitDomProps = function () {
    throw new Error("doInitDom not implemented");
  };
  /**
   * Reads the editor data into a TabularConversion object
   * @returns {Promise} promise to be resolved with the current value
   */


  BaseTabularConversionEditor.prototype.doRead = function () {
    return Error.reject("doRead not implemented");
  };
  /**
   * Build the Tabular Conversion editor comprising of a description and
   * provision to enter 'n' number of points and manage them.
   *
   * @param {jQuery} element the DOM element into which to load this widget
   */


  BaseTabularConversionEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.html(templateEditor(that.getInitDomProps()));
    dom.on('focus', '.editors .point-row', function () {
      var $this = $(this);
      that.$getEditorsElement().children().removeClass('ux-bg');
      $this.addClass('ux-bg');
      that.$focusedEditor = $this.data('widget');
    });
    return Promise.join(fe.buildFor({
      value: '',
      dom: $('.js-desc', dom),
      formFactor: 'mini'
    }), fe.buildFor({
      dom: $('.action-group', dom),
      type: CommandButtonGroup,
      value: this.getCommandGroup()
    }), CompositeEditor.prototype.doInitialize.apply(this, arguments));
  };
  /**
   * Load the widget with a TabularConversion object
   * @param {TabularConversion} value - value to load
   */


  BaseTabularConversionEditor.prototype.doLoad = function (value) {
    return Promise.join(this.$getDescEditor().load(value.getDescription()), CompositeEditor.prototype.doLoad.call(this, value));
  };
  /**
   * Destroy all child editors.
   *
   * @returns {Promise}
   */


  BaseTabularConversionEditor.prototype.doDestroy = function () {
    return this.getChildWidgets().destroyAll();
  };
  /**
   * Set all child editors readonly/writable.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */


  BaseTabularConversionEditor.prototype.doReadonly = function (readonly) {
    return this.getChildWidgets().setAllReadonly(readonly);
  };
  /**
   * Set all child editors enabled/disabled.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */


  BaseTabularConversionEditor.prototype.doEnabled = function (enabled) {
    return this.getChildWidgets().setAllEnabled(enabled);
  }; ///////////////////Private functions///////////////////////////////

  /**
   * Get the editor corresponding to the list of points
   * @private
   * @returns {jQuery}
   */


  BaseTabularConversionEditor.prototype.$getEditorsElement = function () {
    return this.jq().find('.editors');
  };
  /**
   * Get the editor corresponding to the list of points
   * @private
   * @returns {jQuery}
   */


  BaseTabularConversionEditor.prototype.$getContentElement = function () {
    return this.jq().find('.data-table');
  };
  /**
   * @private
   * @returns {module:bajaux/util/CommandButtonGroup}
   */


  BaseTabularConversionEditor.prototype.$getCommandButtonGroup = function () {
    return this.jq().children('.CommandButtonGroup').data('widget');
  };
  /**
   * @private
   * @returns {module:bajaux/commands/Command}
   */


  BaseTabularConversionEditor.prototype.$getAddCommand = function () {
    return this.getCommandGroup().get(0);
  };
  /**
   * @private
   * @returns {module:bajaux/commands/Command}
   */


  BaseTabularConversionEditor.prototype.$getRemoveCommand = function () {
    return this.getCommandGroup().get(3);
  };
  /**
   * @private
   * @returns {module:bajaux/commands/Command}
   */


  BaseTabularConversionEditor.prototype.$getDelAllCommand = function () {
    return this.getCommandGroup().get(1);
  };
  /**
   * @private
   * @returns {module:bajaux/commands/Command}
   */


  BaseTabularConversionEditor.prototype.$getResortCommand = function () {
    return this.getCommandGroup().get(2);
  };
  /**
   * Get the description editor
   * @returns {*} The data widget corresponding to the description
   */


  BaseTabularConversionEditor.prototype.$getDescEditor = function () {
    return this.jq().find(".js-desc").data("widget");
  };
  /**
   * Set the currently focused editor to the for the given index.
   * @private
   * @param {number} i
   */


  BaseTabularConversionEditor.prototype.$setFocusedIndex = function (i) {
    var focusedEd = this.getBuilder().getEditorFor(String(i));
    this.$focusedEditor = focusedEd;
    focusedEd.requestFocus();
  };

  return BaseTabularConversionEditor;
});
//# sourceMappingURL=BaseTabularConversionEditor.js.map
