/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Patrick Sager
 */

/* global sjcl */

/**
 * API Status: **Private**
 * @module nmodule/gauth/rc/GoogleSecretKeyEditor
 */
define(['baja!',
        'baja!baja:Password',
        'lex!gauth',
        'nmodule/webEditors/rc/fe/baja/BaseEditor',
        'nmodule/webEditors/rc/fe/baja/util/compUtils',
        'bajaux/mixin/subscriberMixIn',
        'jquery',
        'Promise',
        'dialogs',
        'nmodule/gauth/ext/qrcode-generator/qrcode',
        'nmodule/webEditors/rc/servlets/password',
        'hbs!nmodule/gauth/rc/template/GoogleSecretKeyEditor',
        'hbs!nmodule/gauth/rc/template/GoogleSecretKeyEditor-confirm',
        'hbs!nmodule/gauth/rc/template/GoogleSecretKeyEditor-qrcode',
        'hbs!nmodule/gauth/rc/template/GoogleSecretKeyEditor-verify',
        'css!nmodule/gauth/rc/gauth',
        'nmodule/gauth/ext/sjcl/sjcl.built.min'], function (
        baja,
        types,
        lexicons,
        BaseEditor,
        compUtils,
        subscriberMixin,
        $,
        Promise,
        dialogs,
        Qrcode,
        password,
        tplGoogleSecretKeyEditor,
        tplGoogleSecretKeyEditorConfirm,
        tplGoogleSecretKeyEditorQrcode,
        tplGoogleSecretKeyEditorVerify) {

  'use strict';

  var lex = lexicons[0];

  function generateToken(key, t) {
    var hmac = new sjcl.misc.hmac(sjcl.codec.base32.toBits(key), sjcl.hash.sha1);

    var hash = hmac.encrypt([t & 0xFFFFFFFF00000000, t << 32]);
    var length = sjcl.bitArray.bitLength(hash);

    var offset = (sjcl.bitArray.bitSlice(hash, length - 8)[0] >>> 24) & 0x0F;
    var truncatedHash = sjcl.bitArray.bitSlice(hash, (offset*8), (offset*8)+32)[0];

    truncatedHash &= 0x7FFFFFFF;
    truncatedHash %= 1000000;
    return truncatedHash;
  }

  /**
   * Editor for generating a Google auth secret key.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/gauth/rc/GoogleSecretKeyEditor
   */
  var GoogleSecretKeyEditor = function GoogleSecretKeyEditor() {
    /** remember to call super constructor. Javascript won't do this for you */
    BaseEditor.apply(this, arguments);
    subscriberMixin(this);
  };

  //extend and set up prototype chain
  GoogleSecretKeyEditor.prototype = Object.create(BaseEditor.prototype);
  GoogleSecretKeyEditor.prototype.constructor = GoogleSecretKeyEditor;

  /**
   * Find the closest ancestor that is a baja:User, lease it if available,
   * and return a promise that resolves with the the user name, or resolve with
   * 'unknown' if no user found.
   *
   * @returns {Promise} to be resolved with the username, or 'unknown'
   */
  GoogleSecretKeyEditor.prototype.$getUserName = function () {
    var complex = this.getComplex();
    var user = complex && compUtils.closest(complex, 'baja:User');

    if (user) {
      return Promise.resolve(user.isMounted() && user.lease())
        .then(function () {
          return user.getName();
        });
    }

    return Promise.resolve('unknown');
  };

  /**
   * Find the closest ancestor that is a baja:Station, lease it if available,
   * and return a promise that resolves with the station name, or resolve with
   * 'unknown' if no station found.
   *
   * @returns {Promise} to be resolved with the station name, or 'unknown'
   */
  GoogleSecretKeyEditor.prototype.$getStationName = function () {
    var that = this;

    return baja.Ord.make("station:|slot:/stationName").get()
      .catch(function (err) {
        var complex = that.getComplex();
        var station = complex && compUtils.closest(complex, 'baja:Station');

        if (station) {
          return Promise.resolve(station.isMounted() && station.loadSlots())
            .then(function () {
              return station.getStationName();
            })
            .catch(function (err) {
              return 'unknown';
            });
        }
        return 'unknown';
      });
  };

  /**
   * If the complex is a gauth:GoogleAuthAuthenticator, make an rpc call
   * to isSecretKeyConfigured and resolve with the results, otherwise
   * resolve with false.
   *
   * @private
   * @returns {Promise} to be resolved with the results of the rpc call
   */
  GoogleSecretKeyEditor.prototype.$isSecretKeyConfigured = function () {
    var complex = this.getComplex();
    if (baja.hasType(complex, 'gauth:GoogleAuthAuthenticator') && complex.getNavOrd()) {
      return baja.rpc(complex.getNavOrd(), 'isSecretKeyConfigured');
    }
    return Promise.resolve(false);
  };

  /**
   * Check to make sure this password editor is operating in a secure
   * environment. Note that this is a client-side check only to prevent editor
   * from validating. If the station detects passwords from an insecure source
   * it must reject those passwords.
   *
   * @private
   * @returns {Promise}
   */
  GoogleSecretKeyEditor.prototype.$isSecure = function () {
    return password.$isSecure();
  };

  /**
   *  initialize the editor
   *
   * @param {jQuery} dom
   * @returns {Promise}
   */
  GoogleSecretKeyEditor.prototype.doInitialize = function (dom) {
    var that = this;

      return that.$isSecretKeyConfigured()
      .then(function (configured) {
        return configured;
      }, function (err) {
        return false;
      })
      .then(function (configured) {
        dom.html(tplGoogleSecretKeyEditor({
          displayName: configured ? lex.get('auth.regenerateKey') : lex.get('auth.generateKey')
        }));

        dom.on('click', 'button.generate', function () {
          var show = configured || that.$secretKey ? that.$showConfirmDialog : that.$generateQr;
          show.call(that).catch(dialogs.showOk);
        });

        return that.$isSecure();
      })
      .then(function (secure) {
        that.jq().find('button.generate').prop('disabled', !secure);
      });
  };

  /**
   * Show a dialog that explains to the user that regenerating the secret
   * key will invalidate devices configured with the existing key, and
   * allow them to confirm.
   *
   * @private
   * @returns {Promise}
   */
  GoogleSecretKeyEditor.prototype.$showConfirmDialog = function () {
    var that = this;

    return dialogs.showOkCancel({
      content: tplGoogleSecretKeyEditorConfirm({
        message: lex.get('auth.confirmRegenerate')
      })
    })
      .ok(function (dlg) {
        return dlg.close()
          .promise()
          .then(function() {
            return that.$generateQr();
          });
      })
      .promise();
  };

  /**
   * Generate a secret key, create a QR code for it, and display it in a dialog
   *
   * @private
   * @returns {Promise}
   */
  GoogleSecretKeyEditor.prototype.$generateQr = function () {

    var that = this;
    sjcl.random.startCollectors();
    var random = sjcl.random.randomWords(3);
    this.$secretKey = sjcl.codec.base32.fromBits(sjcl.bitArray.bitSlice(random, 0, 80));
    return Promise.join(this.$getUserName(), this.$getStationName(), function (username, stationName) {
      var qr = Qrcode(4, 'L');
      qr.addData('otpauth://totp/' + username + '@' + stationName + '?secret=' + that.$secretKey);
      qr.make();
      var tag = qr.createImgTag(4);

      return dialogs.showOkCancel({
        content: tplGoogleSecretKeyEditorQrcode({
          instructions: lex.get('auth.qrInstructions'),
          qrCode: tag,
          key: that.$secretKey + ' - ' + username + ' @ ' + stationName
        })
      })
        .ok(function (dlg) {
          return dlg.close()
            .promise()
            .then(function () {
              return that.$showVerifyDialog(false);
            });
        })
        .cancel(function (dlg) {
          that.$secretKey = undefined;
        })
        .promise();
    });
  };

  /**
   * Show a dialog prompting the user to verify correct setup
   * by entering the token povided by their authenticator.
   *
   * @private
   * @param {Boolean} error displays an error message indicating that the
   * user has entered an incorrect token if true.
   */
  GoogleSecretKeyEditor.prototype.$showVerifyDialog = function (error) {
    var that = this;

    return dialogs.showOkCancel({
      content: tplGoogleSecretKeyEditorVerify({
        instructions: lex.get('auth.confirmToken'),
        isError: error,
        error: lex.get('auth.incorrectToken')
      })
    })
      .ok(function (dlg) {
        return dlg.close()
          .promise()
          .then(function () {
            // If the token that the user supplies matches the token that
            // we generate, that means they have the correct secret key.
            // These tokens are time based and change every 30 seconds, so
            // check against 3 past tokens and 3 future tokens to account
            // for time differences between the user's mobile device and browser.
            var enteredToken = dlg.content().find('input').val();
            var t = Math.floor(Date.now() / 30000);
            var verified = false;
            for (var i = -3; i <= 3 && !verified; i++) {
              var expectedToken = generateToken(that.$secretKey, t+i);
              if (parseInt(enteredToken, 10) === expectedToken) {
                verified = true;
              }
            }

            if (verified) {
              that.setModified(true);
              that.jq().find('button.generate').text(lex.get('auth.regenerateKey'));
            } else {
              return that.$showVerifyDialog(true);
            }
          });
      })
      .cancel(function (dlg) {
        that.$secretKey = undefined;
      })
      .promise();
  };

  /**
   * Set the generate key button disabled if readonly, set it enabled if
   * not readonly, unless the connection is not secure.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  GoogleSecretKeyEditor.prototype.doReadonly = function (readonly) {
    var that = this;
    return that.$isSecure()
      .then(function (secure) {
        if (!readonly && !secure) {
          return;
        }

        that.jq().find('button.generate').prop('disabled', readonly);
      });
  };

  /**
   * Set the generate key button disabled if not enabled, set it enabled if
   * enabled, unless the connection is not secure.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  GoogleSecretKeyEditor.prototype.doEnabled = function (enabled) {
    return this.doReadonly(!enabled);
  };

  /**
   * Returns a baja:Password with the value of the secret key that was generated,
   * or the default password if no key was generated or any of the dialogs were canceled.
   *
   * @returns {baja:Password} the secret key that was generated.
   */
  GoogleSecretKeyEditor.prototype.doRead = function () {
    return baja.$('baja:Password', this.$secretKey);
  };

  /**
   * @param {baja.Simple} pwd the `baja:Password` value to load
   */
  GoogleSecretKeyEditor.prototype.doLoad = function (pwd) {
    this.$secretKey = undefined;
  };

  /**
   * Sets the password slot on the complex to the value of the generated key
   * using the password rpc.
   *
   * @returns {Promise}
   */
  GoogleSecretKeyEditor.prototype.saveToComplex = function (pw, params) {
    if (this.$secretKey) {
      return password.setPassword(this.$secretKey, this.getSlot(), this.getComplex());
    }
    return Promise.resolve();
  };

  return GoogleSecretKeyEditor;
});
