/*
  This example demonstrates how we can create and use properties in a widget.

  - A property consists of a name and a value.
  - A value has to be either a string, number or boolean value.
  - A property is primarily used as a way to configure the widget externally. 
    For example, from the Px Editor.
  - In a Px environment, a property can be animated and the widget can receive updates.
  - Attributes for a property can be added that give hints on how it should be rendered and persisted.
  - A property for a bajaux Widget is very simple; it's far simipler than a Niagara BComponent property.

  If you're viewing this from a Px page, you'll notice that on the right hand side of the screen the
  'num' property is being animated against a `Ramp` component in the Station. If the Web Widget is not updating
  then please ensure the Ramp is enabled.
*/
define(['bajaux/Widget'], function (
        Widget) {

  'use strict';

  var PropertiesWidget = function () {
    var that = this;
    Widget.apply(that, arguments);

    that.properties().add("num", 0);

    // If more than just a name and value is required then add as an object literal...
    that.properties().add({
      name: "foo",
      value: true,
      transient: true,    // a transient attribute means the value for this property is never persisted.
      hidden: false,  // hides the property in an editor.
      readonly: true  // makes editor for the property readonly in an editor.
    });

    // We can also provide a type spec to map the property against a Niagara Type. For instance,
    // here's how an enumerated Niagara Property can be created...
    that.properties().add({
      name: "weekday",
      value: "tuesday", // If it's an enum, we use the tag name.
      typeSpec: "baja:Weekday" // Provide a TypeSpec hint so the editor knows what to decode the 
                               // encoded string into.
    });
  };

  PropertiesWidget.prototype = Object.create(Widget.prototype);
  PropertiesWidget.prototype.constructor = PropertiesWidget;

  PropertiesWidget.prototype.doInitialize = function (jq) {
    jq.html("<pre>Property Changes will appear here...\n</pre>");
  };

  PropertiesWidget.prototype.doChanged = function(name, value) {
    // This will be called everytime a property has changed...
    var jq = this.jq().children("pre");
    jq.text(jq.text() + 
      "Property changed. Name: " + 
      name + 
      " value: " +
      value + 
      "\n");
  };

  return PropertiesWidget;
});
