/*
 * A widget that demonstrates the edit, read and save features of bajaux.
 *
 * This widget also introduces the concept of using Handlebars to inject HTML
 * content for the user interface. You can find out more about Handlebars here...
 *
 * http://handlebarsjs.com/
 *
 * Note that certain module IDs include an exclamation point (!). The '!'
 * character indicates the use of a RequireJS plugin. The plugins used for this
 * module include:
 *
 * - BajaScript (baja!): ensures that BajaScript has fully started
 * - Handlebars (hbs!): import Handlebars templates
 * - Lexicon (lex!): load Lexicon data so the widget can be translated into
 *   different languages
 * - CSS (css!): import a CSS stylesheet for this widget
 */
define(['baja!',
        'bajaux/Widget',
        'bajaux/mixin/subscriberMixIn',
        'bajaux/util/SaveCommand',
        'jquery',
        'Promise',
        'hbs!nmodule/docDeveloper/rc/bajaux/examples/ModifiedWidgetTemplate',
        'lex!docDeveloper',
        'css!nmodule/docDeveloper/rc/bajaux/examples/ModifiedWidgetStyle'], function (
         baja,
         Widget,
         subscriberMixIn,
         SaveCommand,
         $,
         Promise,
         template,
         lexicons) {

  'use strict';

  var lex = lexicons[0];

  function getAmpDom(dom) {
    return dom.find('.example-modified-widget-amplitude');
  }

  function getEnabledDom(dom) {
    return dom.find('.example-modified-widget-enabled');
  }

  function getValueDom(dom) {
    return dom.find('.example-modified-widget-value');
  }

  var ModifiedWidget = function () {
    Widget.apply(this, arguments);
    subscriberMixIn(this);

    // Add a Command that enables the user to save the Widget.
    // There's more on Commands in the other examples.
    this.getCommandGroup().add(new SaveCommand());
  };
  ModifiedWidget.prototype = Object.create(Widget.prototype);
  ModifiedWidget.prototype.constructor = ModifiedWidget;

  /**
   * Update the contents of the DOM in which the Widget is initialized. This
   * function uses the Handlebars template we imported to generate the HTML,
   * using the docDeveloper lexicon to supply it with localized values.
   *
   * This function also sets up jQuery event handlers. By default, handlers
   * registered on the `dom` parameter, like then ones we arm in this function,
   * will be automatically cleaned up when the Widget is destroyed. Any
   * additional handlers (on child elements of the `dom` parameter, say, or on
   * elements outside of this Widget) would need to be cleaned up in
   * `doDestroy()` in order to prevent memory leaks.
   *
   * @param {jQuery} dom
   */
  ModifiedWidget.prototype.doInitialize = function (dom) {
    var widget = this;

    dom.html(template({
      value: lex.get("modifyWidget.value"),
      changeValues: lex.get("modifyWidget.changeValues"),
      amplitude: lex.get("modifyWidget.amplitude"),
      enabled: lex.get("modifyWidget.enabled"),
      loading: lex.get("modifyWidget.loading")
    }));

    // When the user makes a change, mark the widget as modified using setModified(true).
    dom.on('keyup', '.example-modified-widget-amplitude', function () {
      widget.setModified(true);
    });

    dom.on('change', '.example-modified-widget-enabled', function () {
      widget.setModified(true);
    });
  };

  /**
   * When a Ramp is loaded, update the DOM to reflect the Ramp's current values.
   * Also, thanks to `subscriberMixIn`, we can subscribe to changes to the
   * Ramp to ensure that the DOM is always kept up to date.
   *
   * @param {baja.Component} ramp a `kitControl:Ramp` to load
   */
  ModifiedWidget.prototype.doLoad = function (ramp) {
    var widget = this,
        dom = widget.jq(),
        valueDom = getValueDom(dom),
        ampDom = getAmpDom(dom),
        enabledDom = getEnabledDom(dom);

    function update() {
      valueDom.val(ramp.getOutDisplay());
      // Only update the editable DOM if the widget isn't modified.
      if (!widget.isModified()) {
        ampDom.val(ramp.getAmplitudeDisplay());
        enabledDom.prop('checked', ramp.getEnabled());
      }
    }

    // Call update whenever a Property changes
    this.getSubscriber().attach('changed', update);

    // Call update for the first time.
    update();
  };

  /**
   * Resolve an object with the current state of the "enabled" and "amplitude"
   * editors.
   *
   * @returns {Object}
   */
  ModifiedWidget.prototype.doRead = function () {
    var dom = this.jq();
    return {
      enabled: getEnabledDom(dom).is(':checked'),
      amplitude: parseFloat(getAmpDom(dom).val(), 10)
    };
  };

  /**
   * Save the user-entered changes to the loaded Ramp. Note that the parameter
   * to this function is the same as that resolved by doRead().
   *
   * @returns {Promise}
   */
  ModifiedWidget.prototype.doSave = function (readValue) {
    var ramp = this.value(),
        batch = new baja.comm.Batch(),
        promises = [
          ramp.set({ slot: 'enabled',   value: readValue.enabled,   batch: batch }),
          ramp.set({ slot: 'amplitude', value: readValue.amplitude, batch: batch })
        ];

    // Commit the changes in one network call.
    batch.commit();

    // Return a Promise so that the framework knows when the save has completed.
    return Promise.all(promises);
  };

  return ModifiedWidget;
});