/*
  This example demonstrates how we can create and use commands in a widget.

  - A command can be invoked on a widget to do something. 
  - A toggle command can be used to hold state (i.e. something being switched on or off).
  - A command can have flags to hint where it should be shown (i.e. menu bar, tool bar or both).
  - If this widget was embedded in a module and then shown as a registered view on a component...
    - In workbench, the commands would show up in the menu and toolbars 
      (providing the flags are configured correctly).
    - In a browser, the commands would show up in the toolbar.
*/
define(['bajaux/Widget',
        'bajaux/commands/Command',
        'bajaux/commands/ToggleCommand',
        'bajaux/events'], function (
        Widget,
        Command,
        ToggleCommand,
        events) {

  'use strict';

  var CommandWidget = function () {
    var widget = this,
        group;

    Widget.apply(widget, arguments);

    // A command group can hold commands, toggle commands and
    // other command groups.
    group = widget.getCommandGroup();

    // The display name, description and icon for a command
    // is held in a module's lexicon. For instance...
    // 
    // commands.helloWorld.displayName=Hello World
    // commands.helloWorld.icon=module://icons/x16/star.png
    // commands.helloWorld.description=My first command!
    // 
    group.add(new Command({
      module: "docDeveloper",
      lex: "commands.helloWorld",
      func: function () {
        widget.jq().text("Clicked the Hello World Command!");
      }
    }));

    group.add(new ToggleCommand({
      module: "docDeveloper",
      lex: "commands.toggleTest",
      func: function () {
        // Toggle the command everytime it is clicked.
        this.toggle();

        // An alterative to calling 'toggle()' is 'setSelected(boolean)'.

        widget.jq().text("Clicked the Toggle me Command: " + this.isSelected());
      }
    }));

    group.add(new Command({
      module: "docDeveloper",
      lex: "commands.disabledTest",
      func: function () {
        // A command can be enabled/disabled. When this command is invoked,
        // the command will become disabled and hence can't be invoked again!
        this.setEnabled(false);

        widget.jq().text("Clicked the disable Command");
      }
    }));

    var listenCmd = new Command({
      module: "docDeveloper",
      lex: "commands.listenTest",
      func: function () {
        // Do something...
      }
    });

    // Event listeners can be added to commands to listen for when they're invoked.
    listenCmd.on(events.command.INVOKE_EVENT, function () {
      widget.jq().text("The listen command has been invoked");
    });

    group.add(listenCmd);
  };

  CommandWidget.prototype = Object.create(Widget.prototype);
  CommandWidget.prototype.constructor = CommandWidget;

  CommandWidget.prototype.doInitialize = function (jq) {
    jq.text("Please click a Command (the one of the icons on this dialog)!");
  };

  return CommandWidget;
});