/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Vikram N
 */

/* global module*/
/*jshint browser: true, loopfunc: true */

/**
 * @module nmodule/web/rc/util/activityMonitor
 */
(function (root, factory) {
  'use strict';

  /**
   * Loads javascripts synchronously
   * @todo blocking call, work out an alternative.
   * @param {String|Array.<String>} uri(s) of the javascript resource
   * @param {Function} [complete] optional callback for when Javascript has completed loading
   */
  function loadJavaScript(links, complete) {
    var counter = 0,
      head = document.head;
    for (var i = 0; i < links.length; i++) {
      var uri = links[i].uri;
      var hasNative = links[i].hasNative;
      if (hasNative) {
        ++counter;
      } else {
        var req = new XMLHttpRequest();
        req.open('GET', uri, false);
        req.onreadystatechange = function () {
          if (req.readyState === 4) {
            var s = document.createElement("script");
            s.appendChild(document.createTextNode(req.responseText));
            head.appendChild(s);
            ++counter;
          }
        };
        req.send(null);
      }
    }
    if (counter === links.length) {
      complete();
    }
  }

  if (typeof define === 'function' && define.amd) {
    define([ 'Promise', 'nmodule/js/rc/log/Log', 'nmodule/js/rc/polyfills/BroadcastChannel', 'nmodule/js/rc/lex/lex' ], factory);
  } else if (typeof module === 'object' && module.exports) {
    // Node.
    module.exports = factory(require('Promise', 'nmodule/js/rc/log/Log', 'nmodule/js/rc/polyfills/BroadcastChannel', 'nmodule/js/rc/lex/lex'));
  } else {
    // Browser globals (root is window)
    //Synchronously load Promise and BroadcastChannel polyfills
    loadJavaScript([ {
      'uri': '/module/js/rc/polyfills/promise/promise.min.js',
      'hasNative': (typeof root.Promise === 'function')
    }, {
      'uri': '/module/js/rc/polyfills/BroadcastChannel.js',
      'hasNative': (typeof root.BroadcastChannel === 'function')
    } ], function () {
      root.activityMonitor = factory(root.Promise, root.console, root.BroadcastChannel);
    });
  }
}(this, function (Promise, Log, BroadcastChannel, lex) {
  'use strict';

  //////////////////////////Private variables Lex /////////////////////////////////////////////////
  var defaultLexMessages = {
      "activityMonitor.session.timeOut.warningMessage": "Your session will expire within 30 seconds. Click OK to continue."
    },
    webLex,
    warningDialog,
    bc;
  /**
   * A lex object to get lexicon values from the web module when available
   * or fallback to locals defined here
   * @constructor
   */
  var Lex = function () {
  };
  /**
   * Initialize the lex object and keep it ready to get values from
   * @return {Promise}
   */
  Lex.prototype.init = function () {
    //This promise does not have to reject because there is a fallback
    // eslint-disable-next-line promise/avoid-new
    return new Promise(function (resolve) {
      if (lex && typeof lex.module === 'function') {
        lex.module("web")
          .then(function (w) {
            webLex = w;
            resolve();
          })
          .catch(function (ignore) {
            Log.info("Falling back to local lex");
            resolve();
          });
      } else {
        Log.info("Falling back to local lex");
        resolve();
      }
    });
  };

  /**
   * Gets the lexicon value from module when available or fallback to local string refs
   * @param {String} key - the lexicon key
   * @return {String} The lexicon value
   */
  Lex.prototype.get = function (key) {
    var lexVal = defaultLexMessages[key];
    var webLexVal = webLex && webLex.getSafe(key);
    if (webLexVal) {
      lexVal = webLexVal;
    }
    return lexVal;
  };

  //////////////////////////Private variables ActivityMonitor//////////////////////////////////////
  var resetIntervalId,
    warnPopupTimeoutId,
    ACTIVITY_EVENTS = [ 'keypress', 'mousedown', 'touchstart', 'DOMMouseScroll', 'mousewheel' ],
    lastActivityTimestamp,
    hasRequire = typeof require === 'function',
    TIMEOUT_URL = "/timeout",
    prevOnbeforeUnload;

  /**
   * Initialize and create a new Activity Monitor
   *
   * @param {Number} [resetInterval = 60000] - Time in milliseconds to send periodic reset to server {defaults to 60000 or 1 minute}
   * @param {Number} [timeToExpiry = 30000] - Time in milliseconds to warn the user of expiry {defaults to 30000 or 30 seconds}
   * @alias module:nmodule/web/rc/util/activityMonitor
   */
  var ActivityMonitor = function (resetInterval, timeToExpiry) {

    this.$lex = new Lex();

    /**
     * Status variable indicating if the monitor has started or in the keep-alive/pause mode
     * @type {boolean}
     */
    this.$started = false;

    /**
     * Interval at which to periodically send a /timeout to the server
     * @type {number} In millis and defaults to 2 minutes
     */
    this.$callResetInterval = parseFloat(resetInterval) || 60000;

    /**
     * The time (available until actual timeout) to popup a warning about session expiry
     * @type {number} In millis and defaults to 1 minute
     */
    this.$timeToWarnOfExpiry = parseFloat(timeToExpiry) || 30000;

    /**
     * Reference to the bound $activityHandler function so we can
     * remove the event listener later
     * @type {Function}
     */
    this.$boundActivityHandler = this.$activityHandler.bind(this);

    /**
     * Reference to the bound $sendResetNow function so we can
     * remove the event listener later
     * @type {Function}
     */
    this.$boundSendResetNow = this.$sendResetNow.bind(this);

    /**
     * Indicates whether or not we are in the timeout warning period and should
     * send the reset message immediately the next time activity is detected.
     *
     * @type {boolean}
     */
    this.$inWarningPeriod = false;

    /**
     * Pre-fetch dialogs module and prevent session reset
     * @see NCCB-30739, NCCB-31017
     */
    if (hasRequire) {
      require([ 'dialogs' ]);
    }
  };

  /**
   * Starts the activity monitor
   * @returns {Promise}
   */
  ActivityMonitor.prototype.start = function () {
    var that = this;

    // eslint-disable-next-line promise/avoid-new
    var p = new Promise(function (resolve, reject) {
      try {
        lastActivityTimestamp = Date.now();
        that.$addActivityListeners(ACTIVITY_EVENTS);
        that.$started = true;
        //Create a broadcast channel
        if (typeof BroadcastChannel === "function") {
          bc = new BroadcastChannel("activity_channel");
          bc.onmessage = function (ev) {
            if (warningDialog) {
              //reset onbeforeunload
              window.onbeforeunload = prevOnbeforeUnload;
              warningDialog.close();
            }
          };
        }
        resolve();
      } catch (e) {
        that.$started = false;
        that.$destroy();
        Log.error(e.message);
        reject(new Error("There was a problem starting the activity monitor"));
      }
    });

    return Promise.all([ this.$lex.init(), p ])
      .then(function () {
        Log.info("Inactivity monitoring has started");
        return that.$startSendResetInterval();
      })
      .catch(function (err) {
        Log.error(err.message);
      });
  };

  /**
   * Pauses the activity monitor. Listens for events but informs the server to keep
   * session alive by sending an offset 0
   * @returns {Promise}
   */
  ActivityMonitor.prototype.pause = function () {
    var that = this;
    // eslint-disable-next-line promise/avoid-new
    return new Promise(function (resolve, reject) {
      try {
        Log.info("Inactivity monitoring is paused for this view");
        that.$started = false;
        that.$startSendResetInterval();
        resolve();
      } catch (e) {
        Log.error(e.message);
        reject(new Error("There was a problem pausing the activity monitor"));
      }
    });
  };

  /**
   * Returns if the monitor is running
   *
   * @return {boolean} true if the monitor has started and running
   */
  ActivityMonitor.prototype.isStarted = function () {
    return this.$started;
  };

  /**
   * Add an array of document events to listen to. Registers the
   * $activityHandler handler to each.
   *
   * @private
   * @param {Array.<string>} listeners - List of events to respond to
   * @example
   *  $addActivityListeners(["mousedown", "keypress"])
   */
  ActivityMonitor.prototype.$addActivityListeners = function (listeners) {
    var that = this;
    if (listeners) {
      for (var i = 0; i < listeners.length; i++) {
        document.addEventListener(listeners[i], that.$boundActivityHandler, true);
      }
    }
    //Add window unload listener for tab closure
    window.addEventListener("beforeunload", that.$boundSendResetNow, true);
  };

  /**
   * Window unload activity handler. Makes the /timeout call
   * when say, a tab is closed.
   * @private
   */
  ActivityMonitor.prototype.$sendResetNow = function () {
    var that = this;
    var offset = 0;
    try {
      if (that.isStarted()) {
        //calculate offset
        offset = Date.now() - lastActivityTimestamp;
      }
      if (typeof navigator.sendBeacon === "function") {
        navigator.sendBeacon(TIMEOUT_URL, offset);
      } else {
        that.$send("POST", TIMEOUT_URL, offset, false);
      }
    } catch (e) {
      Log.error(e.message);
    }
  };

  /**
   *
   * Remove all listeners passed in (as a list) and de-register the
   * resetExpiry handler.
   *
   * @private
   * @param {Array.<string>} listeners - List of events to stop responding to
   * @example
   *  $removeActivityListeners(["mousedown", "keypress"])
   */
  ActivityMonitor.prototype.$removeActivityListeners = function (listeners) {
    var that = this;
    if (listeners) {
      for (var i = 0; i < listeners.length; i++) {
        document.removeEventListener(listeners[i], that.$boundActivityHandler, true);
      }
    }
    //remove the unload listener
    window.removeEventListener("beforeunload", that.$boundSendResetNow, true);
  };

  /**
   * Sets the last activity time stamp to now, upon a user interaction
   * @private
   * @param {Event} e - Native event generated by the user interaction
   */
  ActivityMonitor.prototype.$activityHandler = function (e) {
    lastActivityTimestamp = Date.now();
    if (this.$inWarningPeriod) {
      this.$inWarningPeriod = false;
      this.$sendResetTimeout();
    }
  };

  /**
   * Destroy any active timers and listeners
   * @private
   * @returns {Promise}
   */
  ActivityMonitor.prototype.$destroy = function () {
    var that = this;
    // eslint-disable-next-line promise/avoid-new
    return new Promise(function (resolve, reject) {
      try {
        clearTimeout(resetIntervalId);
        that.$removeActivityListeners(ACTIVITY_EVENTS);
        if (bc) {
          bc.close();
        }
        resolve();
      } catch (e) {
        Log.error(e.message);
        reject(new Error("Cannot destroy monitor"));
      }
    });

  };

  /**
   * Sets an interval to periodically call /timeout
   * @private
   */
  ActivityMonitor.prototype.$startSendResetInterval = function () {
    var that = this;
    that.$sendResetTimeout();
    resetIntervalId = setTimeout(that.$startSendResetInterval.bind(that), that.$callResetInterval);
  };

  /**
   * Make an XMLHttpRequest
   * @private
   *
   * @param {String} method The http method to use
   * @param {String} url - The url to call
   * @param {String} data - The form data to send with (say a post)
   * @return {Promise} Promise that can be resolved to the response text
   */
  ActivityMonitor.prototype.$send = function (method, url, data, isAsync) {
    // eslint-disable-next-line promise/avoid-new
    return new Promise(function (resolve, reject) {
      var xhr = new XMLHttpRequest();
      xhr.open(method, url, isAsync === undefined ? true : isAsync);
      xhr.setRequestHeader("Content-type", "application/x-www-form-urlencoded");
      xhr.onreadystatechange = function () {
        if (/*DONE*/ this.readyState === 4) {
          if (/*OK*/this.status === 200) {
            Log.info("Timeout in millis", xhr.responseText);
            resolve(xhr.responseText);
          } else {
            reject(new Error("XHR Error " + xhr.statusText));
          }
        }
      };
      xhr.onerror = function () {
        reject(new Error("XHR Error " + xhr.statusText));
      };
      xhr.send(data);
    });
  };

  /**
   * Calls the /timeout servlet with an offset of the last (most recent)
   * client activity.
   * A 0 offset means keep alive (@see .keepAlive)
   * @private
   */
  ActivityMonitor.prototype.$sendResetTimeout = function () {
    var that = this;
    var offset = 0;
    if (this.isStarted()) {
      //calculate offset
      offset = Date.now() - lastActivityTimestamp;
    }
    return that.$send("POST", TIMEOUT_URL, "offset=" + offset)
      .catch(function (err) {
        Log.error(err.message);
      })
      .then(function (responseText) {
        clearTimeout(warnPopupTimeoutId);
        var sessionTimeRemaining = responseText && parseFloat(responseText);
        that.$checkExpiry(sessionTimeRemaining);
      });
  };

  /**
   * Checks if the remaining time is close enough to popup a warning dialog
   * @private
   * @param {Number}remainingTime - The time in millis until session expiry
   */
  ActivityMonitor.prototype.$checkExpiry = function (remainingTime) {
    var that = this;
    //If there is any open dialog close it
    if (!bc && warningDialog) {
      warningDialog.close();
    }
    if (!isNaN(remainingTime)) {
      var timeToWarn = remainingTime - that.$timeToWarnOfExpiry;
      if (timeToWarn > 0) {
        warnPopupTimeoutId = setTimeout(that.$showWarningIfNeeded.bind(that), timeToWarn);
      }
    } else {
      //Most likely the server session is invalid and returned the login html. "/" will redirect
      //to the login page if session is really invalid.
      ActivityMonitor.$redirectTo('/');
    }
  };

  /**
   * @param {string} url the URL to navigate to in the browser
   */
  ActivityMonitor.$redirectTo = function (url) {
    window.onbeforeunload = null;
    window.location.assign(url);
  };

  /**
   * Call /reset and get the current timeout before showing the actual warning dialog
   * @private
   */
  ActivityMonitor.prototype.$showWarningIfNeeded = function () {
    var that = this;
    var offset = 0;
    if (this.isStarted()) {
      //calculate offset
      offset = Date.now() - lastActivityTimestamp;
    }
    that.$send("POST", TIMEOUT_URL, "offset=" + offset)
      .then(function (responseText) {
        var serverExpiryTime = responseText && parseFloat(responseText);
        if (!isNaN(serverExpiryTime) && (serverExpiryTime - that.$timeToWarnOfExpiry) <= 0) {
          that.$showWarning();
        }
      })
      .catch(function (err) {
        Log.error(err.message);
      });
  };

  /**
   * Shows a warning dialog when session is about to expire
   * @private
   */
  ActivityMonitor.prototype.$showWarning = function () {
    var that = this;
    prevOnbeforeUnload = window.onbeforeunload;
    window.onbeforeunload = null; //clear out any onbeforeunloads before expiry
    if (hasRequire) {
      require([ "dialogs" ], function (dialogs) {
        warningDialog = dialogs.showOk({
          content: that.$lex.get("activityMonitor.session.timeOut.warningMessage"),
          private: true
        })
          .ok(function () {
            //Force send signal to server and not timeout
            that.$sendResetTimeout();
            if (bc) {
              bc.postMessage("");
            }
            //reset onbeforeunloads
            window.onbeforeunload = prevOnbeforeUnload;
          });
      });
    } else {
      //Hide the document contents before showing an alert
      var prevState = window.document.body.style.visibility;
      window.document.body.style.visibility = "hidden";
      //timeout allows the visibility to be set before the alert
      setTimeout(function () {
        var confirmed = window.confirm(that.$lex.get("activityMonitor.session.timeOut.warningMessage"));
        /**
         * Chrome has changed the behavior of JavaScript dialogs so that they are
         * non-blocking if the tab is not active, therefore we cannot tell whether
         * a call to window.alert() returned because the user dismissed it or because
         * the tab was not in focus. We are using confirm here instead since it will
         * only return true if the user clicked "OK". This may lead to session timeouts
         * without warning if the tab is not active, but that is better than keeping the
         * session alive without activity.
         */
        window.document.body.style.visibility = prevState;
        //reset onbeforeunloads
        window.onbeforeunload = prevOnbeforeUnload;
        if (confirmed) {
          that.$activityHandler();//soft activity
          that.$sendResetTimeout();
        } else {
          that.$inWarningPeriod = true;
        }
      }, 1);
    }
  };

  /////////////////////////////Public static///////////////////////////////////
  var monitor,
    keepAliveTokens = [];

  /**
   * Utility to make a simple alphanumeric token
   * @return {string}
   */
  function makeToken() {
    var token = [],
      gamut = "abcdefghijklmnopqrstuvwxyz0123456789";
    for (var i = 0; i < 5; i++) {
      token.push(gamut.charAt(Math.floor(Math.random() * gamut.length)));
    }
    var newToken = token.join("");
    return keepAliveTokens.indexOf(newToken) === -1 ? newToken : makeToken();
  }

  /**
   * Initialize the activity monitor by calling start.
   *
   * @return {Promise} Promise on start
   * @example
   * define(['nmodule/web/rc/util/activityMonitor'], function
   * (activityMonitor){
   *    activityMonitor.start();
   * });
   */
  ActivityMonitor.start = function () {

    if (ActivityMonitor.$getLocation() === 'https://workbench/index.html') {
      //no activity monitor required for offline workbench views
      return Promise.resolve();
    }

    if (!monitor) {
      monitor = new ActivityMonitor();
    }
    return keepAliveTokens.length === 0 ? monitor.$destroy()
      .then(function () {
        return monitor.start();
      }) : Promise.resolve();
  };

  /**
   * Get the browser's `window.location.href`.
   * @private
   * @return {String}
   */
  ActivityMonitor.$getLocation = function () {
    return window.location.href;
  };

  /**
   * Keeps the view from setting off the server to invalidate a session if there was
   * no activity.
   * If there is a monitor already running, it just pauses it by sending a 0 last activity offset
   * to the server basically telling it to not invalidate the session
   * Note: Make sure to call activityMonitor.start on your view unload to release the monitor pause
   *
   * @return {Promise.<String>} Promise that resolves to a token
   * @example
   * define(['nmodule/web/rc/util/activityMonitor'], function
   * (activityMonitor){
   *    var token = activityMonitor.keepAlive();
   * });
   */
  ActivityMonitor.keepAlive = function () {
    var prom;
    if (!monitor) {
      monitor = new ActivityMonitor();
    }
    //If there are keep alive tokens just return a new one
    if (keepAliveTokens.length === 0) {
      prom = monitor.$destroy();
    }

    var token = makeToken();
    keepAliveTokens.push(token);

    return prom ? Promise.resolve(prom)
      .then(function () {
        return monitor.pause()
          .then(function () {
            return token;
          });
      }) : Promise.resolve(token);
  };

  /**
   * Release the keep alive token
   *
   * @param {Promise.<String>|String} token Promise that can resolve to a token or the token itself
   * @example
   * define(['nmodule/web/rc/util/activityMonitor'], function (activityMonitor){
   *    var token;
   *    activityMonitor.keepAlive()
   *    .then(function(t){
   *      token = t;
   *    });
   *
   *    //Do something...when keep alive is no longer needed (say unload) release the token
   *    activityMonitor.release(token);
   * });
   * Note: For Bajaux Widget @see .mixinKeepAlive()
   */
  ActivityMonitor.release = function (token) {
    return Promise.resolve(token)
      .then(function (tokenVal) {
        var tInd = keepAliveTokens.indexOf(tokenVal);
        if (tInd >= 0) {
          keepAliveTokens.splice(tInd, 1);
        }
        if (keepAliveTokens.length === 0) {
          return ActivityMonitor.start();
        }
      });
  };

  /**
   * A convenient mixin for a Bajaux widget that sets up and releases keepAlive
   * automatically by calling this method.
   * @param {module:bajaux/Widget} widget the bajaux widget to mix into
   * @example
   * MyWidget(){ //In widget constructor
   *  activityMonitor.mixinKeepAlive(this);
   * }
   */
  ActivityMonitor.mixinKeepAlive = function (widget) {
    var doInitialize = widget.doInitialize, doDestroy = widget.doDestroy, token;
    widget.doInitialize = function () {
      return Promise.resolve(doInitialize.apply(this, arguments))
        .then(function () {
          return ActivityMonitor.keepAlive();
        })
        .then(function (t) {
          token = t;
        });
    };
    widget.doDestroy = function () {
      return Promise.resolve(doDestroy.apply(this, arguments))
        .finally(function () {
          return ActivityMonitor.release(token);
        });
    };
  };

  /**
   * Gets the current activity monitor started or paused
   * @return {module:nmodule/web/rc/util/activityMonitor}
   */
  ActivityMonitor.getMonitor = function () {
    return monitor;
  };

  return ActivityMonitor;
}));
