/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/*global niagara*/

/**
 * API Status: **Private**
 * @module nmodule/report/rc/fe/GridTableContainer
 */
define(['baja!', 'baja!report:CompGridColumn,report:SingleRow,report:BqlGrid,report:NiagaraVirtualComponentGrid,report:NiagaraVirtualBqlGrid,webEditors:ContentDensity', 'hbs!nmodule/report/rc/fe/templates/GridTableContainer', 'bajaux/Widget', 'bajaux/commands/Command', 'bajaux/commands/CommandGroup', 'bajaux/dragdrop/dragDropUtils', 'bajaux/mixin/subscriberMixIn', 'bajaux/mixin/responsiveMixIn', 'Promise', 'jquery', 'underscore', 'nmodule/report/rc/fe/GridColumn', 'nmodule/report/rc/fe/GridPaginationHandler', 'nmodule/report/rc/fe/GridPaginationModel', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/util/htmlUtils', 'nmodule/webEditors/rc/util/ListSelection', 'nmodule/webEditors/rc/wb/table/Table', 'nmodule/webEditors/rc/wb/table/pagination/PaginationModel', 'nmodule/webEditors/rc/wb/table/pagination/PaginationWidget', 'nmodule/webEditors/rc/wb/mixin/ContextMenuSupport', 'nmodule/webEditors/rc/wb/menu/menuUtils', 'css!nmodule/report/rc/report'], function (baja, types, templateView, Widget, Command, CommandGroup, dragDropUtils, subscriberMixIn, responsiveMixIn, Promise, $, _, GridColumn, GridPaginationHandler, GridPaginationModel, fe, htmlUtils, ListSelection, Table, PaginationModel, PaginationWidget, addContextMenuSupport, menuUtils) {
  'use strict';

  var DASHBOARD = 'dashboard',
      forSubject = menuUtils.forSubject;

  function widgetDefaults() {
    return {
      properties: {
        extraRows: {
          value: '[]',
          hidden: true,
          readonly: true,
          dashboard: true
        },
        pageSize: {
          value: PaginationModel.DEFAULT_ROWS_PER_PAGE,
          typeSpec: 'baja:Integer',
          dashboard: true
        },
        mobileWidth: {
          value: 800,
          typeSpec: 'baja:Integer'
        },
        bqlQueryLimit: {
          value: 3000,
          typeSpec: 'baja:Integer'
        },
        hyperlink: {
          value: 'slot:',
          typeSpec: 'baja:Ord'
        },
        dashboard: {
          value: false,
          hidden: true,
          "transient": true,
          readonly: true
        },
        contentDensity: {
          value: "medium",
          typeSpec: "webEditors:ContentDensity",
          dashboard: true
        }
      }
    };
  }
  /**
   * An HTML5 grid table container for `report:BqlGrid` and `report:ComponentGrid`
   * components.
   *
   * @class
   * @extends module:bajaux/Widget
   * @alias module:nmodule/report/rc/fe/GridTableContainer
   * @mixes module:bajaux/mixin/subscriberMixIn
   * @mixes module:bajaux/mixin/responsiveMixIn
   */


  var GridTableContainer = function GridTableContainer(params) {
    var that = this;
    Widget.call(that, {
      params: params,
      defaults: widgetDefaults()
    });
    addContextMenuSupport(that);
    var responsiveObj = {};

    responsiveObj[GridTableContainer.MOBILE_CLASS_NAME] = function (mediaInfo) {
      var maxWidth = that.properties().getValue(GridTableContainer.MOBILE_WIDTH_PROP_NAME);
      return mediaInfo.width <= maxWidth;
    };

    responsiveMixIn(that, responsiveObj);
    subscriberMixIn(that, {
      autoSubscribe: false
    });
  };

  GridTableContainer.prototype = Object.create(Widget.prototype);
  GridTableContainer.prototype.constructor = GridTableContainer;
  /**
   * The name of the extra rows property.
   *
   * @type {string} The property name.
   */

  GridTableContainer.EXTRA_ROWS_PROP_NAME = 'extraRows';
  /**
   * The name of the page size property.
   *
   * @type {string} The property name.
   */

  GridTableContainer.PAGE_SIZE_PROP_NAME = 'pageSize';
  /**
   * The name of the bql query limit property.
   *
   * @type {string}
   */

  GridTableContainer.QUERY_LIMIT_PROP_NAME = 'bqlQueryLimit';
  /**
   * The break point when the grid switches into mobile mode.
   *
   * @type {string}
   */

  GridTableContainer.MOBILE_WIDTH_PROP_NAME = 'mobileWidth';
  /**
   * The hyperlink property.
   *
   * @type {string}
   */

  GridTableContainer.HYPERLINK_PROP_NAME = 'hyperlink';
  /**
   * The mobile class name used when the widget switches to use mobile styling.
   *
   * @type {string}
   */

  GridTableContainer.MOBILE_CLASS_NAME = 'grid-table-mobile';
  /**
   * The density of the table contents per webEditors:ContentDensity enum.
   * 
   * @type {string}
   */

  GridTableContainer.DENSITY_PROP_NAME = 'contentDensity';
  /**
   * Hyperlinks to the specified nav ord.
   *
   * @private
   *
   * @param  {baja.Ord} ord The nav ord to hyperlink to.
   */

  GridTableContainer.prototype.doHyperlink = function (ord) {
    // Only hyperlink if we have a valid ORD to hyperlink to.
    if (ord && ord !== baja.Ord.DEFAULT) {
      ord = baja.Ord.make({
        base: ord,
        child: baja.Ord.make(this.properties().getValue(GridTableContainer.HYPERLINK_PROP_NAME))
      });
      niagara.env.hyperlink(ord.relativizeToSession());
    }
  };

  GridTableContainer.prototype.doInitialize = function (dom, params) {
    dom.addClass('GridTableContainer').html(templateView());
    var that = this,
        sel = new ListSelection(),
        pageSize = Math.max(that.properties().getValue(GridTableContainer.PAGE_SIZE_PROP_NAME), 1),
        paginationDom = dom.find('.reportPagination');
    sel.defaultHandler = _.noop; //No selection operations here
    // Configure the table

    that.$table = new Table({
      selection: sel
    });
    var pWidget = new PaginationWidget();
    var pHandler = this.$paginationHandler = new GridPaginationHandler(),
        pModel = this.$paginationModel = new GridPaginationModel({
      rowsPerPage: pageSize,
      handler: pHandler
    });
    pModel.on('changed', function (key, value) {
      if (key === 'rowsPerPage') {
        that.properties().setValue(GridTableContainer.PAGE_SIZE_PROP_NAME, value);
      }
    }); // Arm drag and drop handlers if we're running in dashboard mode.

    if (that.properties().getValue(DASHBOARD)) {
      // Arm dom events
      dom.on('dragover', function (e) {
        e.preventDefault();
      }).on('drop', function (e) {
        that.$updateFromDrop(e.originalEvent.dataTransfer);
        e.preventDefault();
        e.stopPropagation();
      });
    }

    dom.on('dblclick', 'tbody', function (e) {
      return that.doHyperlink(that.$getRowSubjectOrd(e));
    });
    htmlUtils.contextMenuOnLongPress(dom, {
      selector: 'td',
      fireClick: true
    }); // Initialize the table.

    return Promise.all([that.$table.initialize(dom.find('.gridTable'), params), fe.buildFor({
      dom: paginationDom,
      value: pModel,
      formFactor: 'mini'
    }, pWidget)]);
  };
  /**
   * Asynchronously adds data to a grid model for a `report:NiagaraVirtualBqlGrid` component.
   *
   * @private
   *
   * @param {baja.Component} grid The `report:BqlGrid` Component.
   * @param model The grid model that data will be appended too.
   *
   * @returns {Promise} A promise that resolves to a grid model.
   */


  GridTableContainer.prototype.$resolveNiagaraVirtualBqlModel = function (grid, model) {
    return grid.rpc('resolveQuery').then(function (result) {
      // Add all the columns.
      model.columns = model.columns.concat(result.columns.map(function (col) {
        return new GridColumn(col.name, {
          displayName: col.displayName,
          format: baja.Format.make('%' + col.name + '%')
        });
      })); // Add each row.

      model.rows = model.rows.concat(result.rows.map(function (ord) {
        return {
          ord: baja.Ord.make(ord)
        };
      }));
    });
  };
  /**
   * Asynchronously adds data to a grid model for a `report:BqlGrid` component.
   *
   * A BQL query is resolved to create the grid model.
   *
   * @private
   *
   * @param {baja.Component} grid The `report:BqlGrid` Component.
   * @param model The grid model that data will be appended too.
   *
   * @returns {Promise} A promise that resolves to a grid model.
   */


  GridTableContainer.prototype.$resolveBqlModel = function (grid, model) {
    var that = this,
        ord = grid.getQuery();

    if (ord === baja.Ord.DEFAULT) {
      return Promise.resolve();
    }

    var ordStr = String(ord);

    if (/bql:select \*/.exec(ordStr)) {
      return Promise.reject(new Error('Query must include columns'));
    }

    return baja.Ord.make(ordStr.replace('bql:select', 'bql:select ordInSession,')).get({
      base: grid
    }).then(function (table) {
      var tableColumns = table.getColumns(),
          ordCol = tableColumns[0];

      for (var i = 1; i < tableColumns.length; ++i) {
        model.columns.push(new GridColumn(tableColumns[i].getName(), {
          displayName: tableColumns[i].getDisplayName(),
          format: baja.Format.make('%' + tableColumns[i].getName() + '%')
        }));
      }

      return table.cursor({
        // Max number of query results in BQL result. With a page size
        // of 20 this 150 button clicks so it's probably enough.
        limit: Math.max(that.properties().getValue(GridTableContainer.QUERY_LIMIT_PROP_NAME), 1),
        each: function each(cursor) {
          model.rows.push({
            ord: cursor.get(ordCol.getName())
          });
        }
      });
    });
  };
  /**
   * Builds the intermediate model that transforms to a TableModel
   * post resolution of all ords
   * model.columns = [Column]
   * model.rows = [{ord: ord, additionalOrds: {}}]
   */


  function resolveComponentModel(baseOrd, grid, model) {
    var ords = {};
    grid.getSlots().dynamic().each(function (slot) {
      if (slot.getType().is('report:CompGridColumn')) {
        var colVal = slot.getDefaultValue();
        model.columns.push(new GridColumn(slot.getName(), {
          ord: colVal.getOrd(),
          displayName: colVal.get('displayName'),
          format: colVal.getFormat()
        }));
      } else if (slot.getType().is('report:SingleRow')) {
        var ord = baja.Ord.make({
          base: baseOrd,
          child: grid.get(slot).getOrd()
        }).normalize(),
            ordStr = String(ord),
            ordObj = {
          ord: ord
        };
        model.rows.push(ordObj);
        ords[ordStr] = ordStr;
      }
    }); // Calculate any additional ORDs that need to be resolved for each row.
    // Since a column can define its own slot path, we could end up with a different
    // target component per cell. Therefore we need to make sure we resolve and subscribe
    // additional ORDs when we load the page.

    model.rows.forEach(function (row) {
      row.additionalOrds = {};
      model.columns.forEach(function (column) {
        try {
          var newOrd = baja.Ord.make({
            base: row.ord,
            child: column.getOrd()
          }).normalize(),
              ordStr;

          if (!newOrd.equals(row.ord)) {
            ordStr = String(newOrd);
            row.additionalOrds[ordStr] = newOrd;
            ords[ordStr] = ordStr;
          }
        } catch (ignore) {}
      });
    });
    return ords;
  }
  /**
   * Asynchronously adds data to a grid model for a `report:ComponentGrid` component.
   *
   * Child `report:CompGridColumn` and `report:SingleRow`
   * structs are scanned for as child properties to create the grid model.
   *
   * @private
   *
   * @param  {baja.Component} grid The `report:BqlGrid` Component.
   * @param model The grid model that data will be appended too.
   * @returns {Promise} A promise that resolves to a grid model.
   */


  GridTableContainer.prototype.$resolveComponentModel = function (grid, model) {
    resolveComponentModel(grid.getNavOrd(), grid, model);
    return Promise.resolve();
  };
  /**
   * Asynchronously adds data to a grid model for a `report:NiagaraVirtualComponentGrid` component.
   *
   * Child `report:CompGridColumn` and `report:SingleRow`
   * structs are scanned for as child properties to create the grid model. The ords are
   * then translated into virtual ords.
   *
   * @private
   *
   * @param  {baja.Component} grid The `report:BqlGrid` Component.
   * @param model The grid model that data will be appended too.
   * @returns {Promise} A promise that resolves to a grid model.
   */


  GridTableContainer.prototype.$resolveNiagaraVirtualComponentModel = function (grid, model) {
    var ords = resolveComponentModel(grid.getNiagaraVirtualCompInfo().getSlotOrd(), grid, model); // Convert from ords to virtual ords.

    return grid.rpc('convertOrdsToVirtualOrds', _.keys(ords)).then(function (ordsToVirtualOrds) {
      model.rows.forEach(function (obj) {
        var virtualOrd = ordsToVirtualOrds[String(obj.ord)];

        if (virtualOrd) {
          obj.ord = baja.Ord.make(virtualOrd);
        }

        if (obj.additionalOrds) {
          var newAdditionalOrds = {};

          _.each(obj.additionalOrds, function (addOrd, addOrdStr) {
            var addVirtualOrd = ordsToVirtualOrds[addOrdStr];

            if (addVirtualOrd) {
              newAdditionalOrds[addVirtualOrd] = baja.Ord.make(addVirtualOrd);
            } else {
              newAdditionalOrds[addOrdStr] = addOrd;
            }
          });

          obj.additionalOrds = newAdditionalOrds;
        }
      });
    });
  };

  GridTableContainer.prototype.doLoad = function (grid) {
    var that = this,
        model = {
      columns: [],
      rows: JSON.parse(that.properties().getValue(GridTableContainer.EXTRA_ROWS_PROP_NAME, '[]'))
    }; // If extra rows property is defined, then make ords.

    if (model.rows.length > 0) {
      model.rows = model.rows.map(function (row) {
        return {
          ord: baja.Ord.make(row.ord)
        };
      });
    }

    return grid.lease().then(function () {
      var typeSpec = String(grid.getType());

      switch (typeSpec) {
        case 'report:BqlGrid':
          return that.$resolveBqlModel(grid, model);

        case 'report:ComponentGrid':
          return that.$resolveComponentModel(grid, model);

        case 'report:NiagaraVirtualComponentGrid':
          return that.$resolveNiagaraVirtualComponentModel(grid, model);

        case 'report:NiagaraVirtualBqlGrid':
          return that.$resolveNiagaraVirtualBqlModel(grid, model);

        default:
          throw new Error('Unsupported grid: ' + typeSpec);
      }
    }).then(function () {
      var params = _.extend({
        subscriber: that.getSubscriber()
      }, model);

      that.$paginationModel.setRowCount(model.rows.length);
      that.$paginationHandler.setParams(params); // Set the table's density

      that.$table.properties().setValue('density', that.properties().getValue('contentDensity'));
      return that.$table.load(that.$paginationModel);
    });
  };

  GridTableContainer.prototype.doDestroy = function () {
    this.jq().removeClass('GridTableContainer');
    return this.$table.destroy();
  };
  /**
   * Called when a new data value is dragged and dropped onto the widget.
   *
   * @private
   *
   * @param {DataTransfer} dataTransfer The transfer data.
   * @return {Promise} A promise that's resolved once the drag and drop operation
   * has completed.
   */


  GridTableContainer.prototype.$updateFromDrop = function (dataTransfer) {
    var that = this;
    return dragDropUtils.fromClipboard(dataTransfer).then(function (envelope) {
      switch (envelope.getMimeType()) {
        case 'niagara/navnodes':
          return envelope.toJson().then(function (json) {
            if (!json || !json.length) {
              return;
            }

            var rows = JSON.parse(that.properties().getValue(GridTableContainer.EXTRA_ROWS_PROP_NAME, '[]'));
            rows = json.map(function (obj) {
              return _.pick(obj, 'ord');
            }).concat(rows);
            that.properties().setValue(GridTableContainer.EXTRA_ROWS_PROP_NAME, JSON.stringify(rows));
            return that.doLoad(that.value());
          });
      }
    })["catch"](baja.error);
  }; ////////////////////////////////////////////////////////////////
  // Context menu support
  ////////////////////////////////////////////////////////////////

  /**
   * Callback from CommandGroupContextMenu.
   *
   * Trigger context menus specifically on table cell right clicks.
   *
   * @override
   * @returns {string}
   */


  GridTableContainer.prototype.getContextMenuSelector = function () {
    return 'td';
  };
  /**
   * Callback from ContextMenuSupport. Returns array of currently selected
   * subjects for context menu or an empty array for no subjects.
   *
   * @override
   * @returns {Promise}
   */


  GridTableContainer.prototype.getSubject = function (elem) {
    var colIndex = elem.index(),
        rowIndex = elem.closest('tr').index(),
        tableModel = this.$table.getModel(),
        row = tableModel.getRows()[rowIndex],
        col = tableModel.getColumns()[colIndex],
        target = row.getSubject().data[col.getName()].object;

    if (target && baja.hasType(target, "baja:Component")) {
      return Promise.resolve([target]);
    }

    return Promise.resolve([]);
  };
  /**
   * Callback to implement a right click or long press event 
   * to generate context menu
   * 
   * @private
   * @param {Event} e - The context menu event
   * @returns {Promise}
   */


  GridTableContainer.prototype.$toContextMenu = function (e) {
    var _this = this;

    return Promise.resolve(this.getSubject($(e.target))).then(function (subject) {
      if (!subject) {
        return;
      }

      return forSubject(_this, subject).then(function (group) {
        if (!group) {
          return;
        }

        return _this.$makeMenuCommandGroup(group, _this.$getRowSubjectOrd(e));
      });
    });
  };
  /**
   * Update the command group prior to displaying the menu.
   *
   * @private
   * @param {module:bajaux/commands/CommandGroup} group 
   * @param {baja.Ord} gotoOrd
   * @returns {Promise}
   */


  GridTableContainer.prototype.$makeMenuCommandGroup = function (group, gotoOrd) {
    // Find Actions group and create a new group.
    var that = this;
    var newGroup = new CommandGroup(),
        actionsGroup = group.getChildren().filter(function (grp) {
      return grp.getDisplayNameFormat().indexOf("webEditors:menu.actions.label") >= 0;
    })[0];

    if (actionsGroup) {
      actionsGroup.getChildren().forEach(function (kid) {
        newGroup.add(kid);
      });
    }

    newGroup.add(new Command({
      module: 'report',
      lex: 'ComponentGridTable.hyperlinkCommand',
      func: function func() {
        return that.doHyperlink(gotoOrd);
      }
    }));
    return Promise.resolve(newGroup);
  };
  /**
   * Get the ord associated with the row's subject
   * 
   * @private
   * @param {Event} e
   * @returns {baja.Ord}
   */


  GridTableContainer.prototype.$getRowSubjectOrd = function (e) {
    var tableModel = this.$table.getModel(),
        colIndex = $(e.target).closest('td').index(),
        rowIndex = $(e.target).closest('tr').index(),
        row = tableModel.getRows()[rowIndex],
        col = tableModel.getColumns()[colIndex],
        rowOrd = row.getSubject().ord,
        target = col.getValueFor(row);

    if (target && typeof target.getNavOrd === 'function') {
      rowOrd = target.getNavOrd();
    }

    return rowOrd;
  };
  /**
   * Detect contentDensity property change on the grid table and set the 
   * underlying "Table" density property. 
   * 
   * @param {string} propName name of the property
   * @param {Number} value value of the property to set
   */


  GridTableContainer.prototype.doChanged = function (propName, value) {
    if (propName !== 'contentDensity') {
      return;
    }

    this.$table.properties().setValue('density', value);
  };

  return GridTableContainer;
});
