/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * API Status: **Private**
 * @module nmodule/report/rc/fe/GridPaginationHandler
 */
define(['baja!', 'underscore', 'Promise', 'nmodule/webEditors/rc/wb/table/model/TableModel', 'nmodule/report/rc/fe/GridColumn'], function (baja, _, Promise, TableModel, GridColumn) {
  'use strict';
  /**
   * Handles pagination in a grid table.
   *
   * @class
   * @alias module:nmodule/report/rc/fe/GridPaginationHandler
   */

  var GridPaginationHandler = function GridPaginationHandler() {};
  /**
   * Pagination parameters.
   *
   * @typedef {Object} paginationParameters
   * @property {baja.Subscriber} subscriber The subscriber used for
   * subscribing/unsubscribing components.
   * @property {Array<module:nmodule/report/rc/fe/GridColumn>} columns An array of
   * columns.
   * @property {Array<Object>} rows An array of row objects that have an 'ord' property.
   */

  /**
   * Sets some parameters for the pagination handler.
   *
   * @param {paginationParameters} params An object that contains parameters.
   */


  GridPaginationHandler.prototype.setParams = function (params) {
    this.$params = {
      subscriber: params.subscriber,
      rows: params.rows.slice(0),
      columns: params.columns.slice(0)
    };
  };
  /**
   * Perform any finalize activities here.
   *
   * @override
   */


  GridPaginationHandler.prototype.destroy = function () {
    this.$params = null;
  };
  /**
   * The data used for updating the pagination.
   *
   * @typedef {Object} paginationData
   * @property {Number} noOfPages The number of pages that can be displayed.
   * @property {Number} pageIndex The index of the page.
   * @property {Number} startIndex The start index of the rows to display on a page (zero indexed).
   * @property {Number} endIndex The end index of the rows to display on the page (zero indexed).
   */

  /**
   * Return the data used for pagination based on the current parameters.
   *
   * @private
   *
   * @param {Number} pageSize The maximum number of entries on a single page.
   * @param {Number} pageIndex  The index of the page to load (starts at 1, not zero based).
   * @returns {paginationData} The pagination data.
   */


  GridPaginationHandler.prototype.$getPaginationData = function (pageSize, pageIndex) {
    var noOfPages = Math.ceil(this.$params.rows.length / pageSize),
        newPageIndex = pageIndex && pageIndex > noOfPages ? noOfPages : pageIndex || 1,
        startIndex = (pageIndex - 1) * pageSize;
    return {
      noOfPages: noOfPages,
      pageIndex: newPageIndex,
      startIndex: startIndex,
      endIndex: startIndex + pageSize
    };
  };
  /**
   * Resolves the components that will be displayed on a page.
   *
   * @private
   *
   * @param {Array} ords All the ords to batch resolve.
   * @param {Array} effectiveRows The temp row model to process for this page.
   * @returns {Promise<Array<{}>>} A promise that resolves to an array of objects.
   */


  GridPaginationHandler.prototype.$resolveRowComponents = function (ords, effectiveRows) {
    var that = this,
        br = new baja.BatchResolve(ords);

    function processRows() {
      return effectiveRows.map(function (row) {
        var rowObj = {
          ord: row.ord,
          data: {}
        };
        var ordToMatch = row.ord.toString(); // Check if working with handles (like BqlGrid) or virtuals

        var ordQueryList = row.ord.parse(),
            isHandle = ordQueryList.get('h') !== null,
            isVirtual = ordQueryList.get('virtual') !== null;
        that.$params.columns.map(function (col) {
          if (!isHandle || !_.isEmpty(row.additionalOrds)) {
            if (isVirtual) {
              ordToMatch = normalizeVirtual(row.ord, col.getOrd());
            } else {
              ordToMatch = baja.Ord.make({
                base: row.ord,
                child: col.getOrd()
              }).normalize().toString();
            }
          }

          var selectedIndex = -1;

          var target = _.find(ords, function (ord, i) {
            if (!br.isResolved(i)) {
              return false;
            } else {
              selectedIndex = i;
              return ord.toString() === ordToMatch;
            }
          });

          if (target) {
            var targetObj = br.getTarget(selectedIndex).getObject();
            rowObj.data[col.getName()] = {
              object: targetObj,
              value: col.$format.format({
                object: targetObj
              })
            };
          } else {
            rowObj.data[col.getName()] = {
              object: null,
              value: GridColumn.NO_VALUE
            };
          }
        });
        return rowObj;
      });
    }

    return br.resolve({
      subscriber: that.$params.subscriber
    }).then(processRows, processRows);
  };
  /**
   * @private
   * @param {number} rowStart
   * @param {number} rowEnd
   * @returns {Promise<module:nmodule/webEditors/rc/wb/table/model/TableModel>}
   */


  GridPaginationHandler.prototype.$resolveRange = function (rowStart, rowEnd) {
    var effectiveRows = this.$params.rows.slice(rowStart, rowEnd); // Adapt {ord:ord, additionalOrds:{}} to a Row {ord:ord, data:{}}

    return this.$tempRowsToTableModel(effectiveRows);
  };
  /**
   * Fetches new page results for the passed in page index
   *
   * @private
   * @override
   *
   * @param {Table} table The source table
   * @param {Number} pageIndex The page index to change to
   * @returns {module:nmodule/webEditors/rc/wb/table/model/TableModel|Promise} new
   * table model, or promise that resolves to one
   */


  GridPaginationHandler.prototype.pageChanged = function (table, pageIndex) {
    var that = this,
        pWidget = table.$getPaginationWidget(),
        pageSize = pWidget.getPageSize(),
        pageData = that.$getPaginationData(pageSize, pageIndex); // Adapt {ord:ord, additionalOrds:{}} to a Row {ord:ord, data:{}}

    return that.$resolveRange(pageData.startIndex, pageData.endIndex).then(function (newTableModel) {
      // Call this again since the number of pages may have changes post ORD
      // resolution.
      var pageData = that.$getPaginationData(pageSize, pageIndex);
      pWidget.updateWidgetDisplay({
        currentIdx: pageData.pageIndex,
        noOfPages: pageData.noOfPages
      });
      return newTableModel;
    });
  };
  /**
   * Transform the temporary model structure to a TableModel
   *
   * @param {Array} effectiveRows The temp row model to process for this page.
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/model/TableModel>} The
   * actual table data consumed by Table
   */


  GridPaginationHandler.prototype.$tempRowsToTableModel = function (effectiveRows) {
    var that = this,
        ords = effectiveRows.map(function (row) {
      return row.ord;
    }),
        subscriber = that.$params.subscriber; // Add on any additional ORDs to be resolved. These aren't used for rows but we
    // need to resolve and subscribe to them regardless.

    effectiveRows.forEach(function (row) {
      if (row.additionalOrds) {
        _.each(row.additionalOrds, function (ord) {
          ords.push(ord);
        });
      }
    });
    return subscriber.unsubscribeAll().then(function () {
      return that.$resolveRowComponents(ords, effectiveRows);
    }).then(function (rows) {
      var tableModel = new TableModel({
        rows: rows,
        columns: that.$params.columns
      });
      subscriber.detach('changed'); //TODO: this needs to recalculate the data for the appropriate rows

      subscriber.attach('changed', _.throttle(function () {
        tableModel.emit('rowsChanged', tableModel.getRows());
      }, 1000, {
        leading: false
      }));
      return tableModel;
    });
  };

  function normalizeVirtual(base, child) {
    if (!child) {
      return String(base);
    }

    var childSlotPath = child.normalize().parse().get('slot');
    var childOrdStr = String(child);

    if (childSlotPath) {
      var childPathBody = childSlotPath.getBody();
      childOrdStr = childPathBody !== "" ? "/" + childPathBody : childPathBody;
      return String(base) + childOrdStr;
    }

    return baja.Ord.make({
      base: base,
      child: child
    }).normalize().toString();
  }

  return GridPaginationHandler;
});
