/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * API Status: **Private**
 * @module nmodule/report/rc/fe/GridColumn
 */
define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/wb/table/model/Column', 'nmodule/webEditors/rc/wb/mgr/componentStatusUtils'], function (baja, Promise, _, Column, componentStatusUtils) {
  'use strict';

  var SLOT_ORD = baja.Ord.make('slot:');
  /**
   * A column for a grid table.
   *
   * @class
   * @alias module:nmodule/report/rc/fe/GridColumn
   * @extends module:nmodule/webEditors/rc/wb/table/model/Column
   *
   * @param {String} name The name of the column
   * @param {Object} [params] Additional parameters for the column.
   * @param {baja.Format} [params.format] Formatter used to display value string.
   * @param {baja.Ord} [params.ord] The column's ORD.
   */

  var GridColumn = function GridColumn(name, params) {
    params = params || {};
    Column.call(this, name, params);
    this.$format = params.format || baja.Format.DEFAULT;
    this.$ord = params.ord || SLOT_ORD;
    this.setSortable(false);
  };

  GridColumn.prototype = Object.create(Column.prototype);
  GridColumn.prototype.constructor = GridColumn;
  /**
   * A symbol of used when there is no value.
   *
   * @type {String}
   */

  GridColumn.NO_VALUE = '-';
  /**
   * The attribute name for the data cell prefix.
   *
   * @type {String}
   */

  GridColumn.DATA_CELL_PREFIX_ATTR = 'data-cell-prefix';
  /**
   * The attribute name for the title.
   *
   * @type {String}
   */

  GridColumn.TITLE_ATTR = 'title';

  GridColumn.prototype.getValueFor = function (row) {
    return row.getSubject().data[this.getName()].value;
  };

  GridColumn.prototype.buildCell = function (row, dom) {
    var that = this,
        obj = row.getSubject().data[this.getName()].object,
        value = row.getSubject().data[this.getName()].value;
    var display = value === GridColumn.NO_VALUE ? value : that.$format.format({
      object: obj
    });
    return Promise.all([that.toDisplayName(), display]).then(function (params) {
      var displayName = params[0] || '',
          display = params[1] || ''; // Used when the table is collapsed for mobile.

      dom.attr(GridColumn.DATA_CELL_PREFIX_ATTR, displayName + ': ');

      if (display !== GridColumn.NO_VALUE) {
        // Displays the value when the mouse over happens.
        dom.attr(GridColumn.TITLE_ATTR, display);
        var status = baja.Status.getStatusFromIStatus(obj);
        var cssClass = componentStatusUtils.getStatusCssClass(status);
        dom.addClass(cssClass);
      }

      return dom.text(display);
    });
  };

  GridColumn.prototype.getOrd = function () {
    return this.$ord;
  };

  return GridColumn;
});
