/**
 * @copyright 2012, Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/*jshint devel: true */
/* eslint-env browser */

/**
 * API Status: **Private**
 *
 * A bajaux Container for Hx.
 *
 * @module nmodule/hx/rc/container/hxContainer
 */
define([ "require",
    "baja!",
    "log!nmodule.hx.rc.container.hxContainer",
    "jquery",
    "Promise",
    "hx",
    "dialogs",
    "underscore",
    "bajaux/events",
    "bajaux/container/util",
    "bajaux/dragdrop/dragDropUtils",
    "bajaux/commands/CommandGroup",
    "bajaux/util/CommandButtonGroup",
    "nmodule/webEditors/rc/fe/fe",
    'bajaux/Properties',
    'bajaux/Widget',
    'nmodule/js/rc/asyncUtils/asyncUtils',
    "nmodule/webEditors/rc/fe/baja/util/ComplexDiff",
    "nmodule/webEditors/rc/wb/commands/PopOutCommand",
    'nmodule/webEditors/rc/fe/baja/util/facetsUtils',
    'nmodule/webEditors/rc/util/htmlUtils',
    "lex!hx,bajaux",
    "hbs!bajaux/container/error",
    "nmodule/webEditors/rc/wb/profile/profileUtils" ], function (require,
                                              baja,
                                              log,
                                              $,
                                              Promise,
                                              hx,
                                              dialogs,
                                              _,
                                              events,
                                              util,
                                              dragDropUtils,
                                              CommandGroup,
                                              CommandButtonGroup,
                                              fe,
                                              Properties,
                                              Widget,
                                              asyncUtils,
                                              ComplexDiff,
                                              PopOutCommand,
                                              facetsUtils,
                                              htmlUtils,
                                              lexs,
                                              errorTemplate,
                                              profileUtils) {

    "use strict";

    var exports,
      registered = [],
      hxLex = lexs[0],
      bajauxLex = lexs[1],
      setValueParam = "ext",
      setMetadataParam = "ext",
      hideCommandBar = util.hideCommandBar,
      escapeHtml = htmlUtils.escapeHtml,
      debounceAsync = asyncUtils.debounceAsync,

      UNRECOVERABLE_FAIL_EVENTS = [
        events.INITIALIZE_FAIL_EVENT,
        events.LOAD_FAIL_EVENT
      ],

      RUNTIME_FAIL_EVENTS = [
        events.ENABLE_FAIL_EVENT,
        events.DISABLE_FAIL_EVENT,
        events.DESTROY_FAIL_EVENT,
        events.READONLY_FAIL_EVENT,
        events.WRITABLE_FAIL_EVENT,
        events.SAVE_FAIL_EVENT
        //        events.command.FAIL_EVENT //SAVE_FAIL_EVENT and command FAIL_EVENT double up.
      ],
      saveErrorMessage = null,
      isSavePending = false,
      logError = log.severe.bind(log),
      logInfo = log.info.bind(log),
      getViewQuery = profileUtils.getViewQuery,
      uriToOrd = profileUtils.uriToOrd;

    function canProfileBrowser() {
      return window.niagara && window.niagara.env && window.niagara.env.profileBrowser;
    }


    /*
     * Either destroy all registered widgets or delete the set of widgets pass in.
     * @param {Array.<module:bajaux/Widget>} [widgets]  Optional array of widgets
     * @returns {Promise}
     */
    function destroyWidgets(selectedWidgets) {
      if (selectedWidgets !== undefined) {
        return Promise.map(selectedWidgets, function (widget) {
          return destroyWidget(widget);
        });
      } else {
        return Promise.map(registered, function (obj) {
          return destroyWidget(obj.widget, obj.id);
        });
      }
    }

    /**
     * If the widget is registered call 'cleanup', if the widget is not, just destroy.
     * @param {module:bajaux/Widget} widget
     * @param {Number} [id]
     * @returns {Promise}
     */
    function destroyWidget(widget, id) {

      if (id === undefined) {
        for (var i = 0; i < registered.length; ++i) {
          if (widget === registered[i].widget) {
            id = registered[i].id;
          }
        }
      }

      if (id !== undefined) {
        var jq = widget.jq(),
          containerJq = jq.closest(".bajaux-container"),
          toolbarJq = containerJq.children(".bajaux-toolbar-outer").children(".bajaux-toolbar");
        return cleanup(containerJq, jq, toolbarJq, id);
      }

      return widget.destroy();
    }

    // Destroy the widget when BajaScript shuts down
    baja.preStop(function () {
      destroyWidgets().catch(baja.error);
    });

////////////////////////////////////////////////////////////////
// Error
////////////////////////////////////////////////////////////////

    function error(jq, widget, type, err) {
      var errorJq,
        params;
      baja.error(err);

      if (!widget) {
        widget = Widget.in(jq);
      }

      // Hide the widget
      jq.hide();

      // Remove the bar completely
      jq.parent()
        .parent()
        .children(".bajaux-toolbar-outer")
        .children(".bajaux-toolbar")
        .empty();

      // Find the error DOM
      errorJq = jq.parent().parent().children(".bajaux-error");

      var errString = String(err),
        stack;

      if (err.stack) {
        stack = String(err.stack);
        //Chrome puts a toString of the error at the top of the stack, FF doesn't
        if (stack.substr(0, errString.length) !== errString) {
          stack = errString + '\n' + stack;
        }
      } else {
        stack = errString;
      }

      params = {
        title: bajauxLex.get("error.title"),
        message: bajauxLex.get("error.message"),
        details: bajauxLex.get("error.details"),
        type: type,
        moduleName: widget ? widget.$moduleName : "",
        keyName: widget ? widget.$keyName : "",
        stack: stack
      };

      // Remove any existing error messages
      errorJq.empty();
      errorJq.html(errorTemplate(params));
      errorJq.show();
    }

    /**
     * Handles recoverable errors that are triggered.
     *
     * @param {jQuery.Event} e
     * @param {module:bajaux/Widget} sourceWidget
     * @param {Error} err
     */
    function handleRecoverableError(e, sourceWidget, err) {
      var isBaseWidget = false;
      for (var i = 0; i < registered.length; ++i) {
        if (registered[i].widget === sourceWidget) {
          isBaseWidget = true;
        }
      }
      if (isBaseWidget) {
        baja.error(err);
        var errorMessage = err instanceof Error ? err.message : err;
        if (isSavePending) {
          updateSaveErrorMessage(errorMessage);
        } else {
          showErrorDialog(errorMessage);
        }
      }
    }

    /**
     * Adds save error if not already in array to be displayed at end of save.
     * @param {string} error the error message
     */
    function updateSaveErrorMessage(error) {
      saveErrorMessage = error;
    }

    /**
     * Shows an error dialog with provided message.
     * @param {string} msg
     */
    function showErrorDialog(msg) {
      dialogs.showOk({
        title: hxLex.get('error.title'),
        content: '<span class="js-dialog-wrapTextContent">' + escapeHtml(msg) + '</span>'
      });
    }

    function isError() {
      var error = false;
      $(".bajaux-container > .bajaux-error").each(function () {
        if ($(this).css("display") === "block") {
          error = true;
        }
      });
      return error;
    }

    function errorHandler(jq, widget, type, err) {
      error(jq, widget, type, err);
      throw err;
    }

    function armFailureHandlers(jq, widget) {

      var failureEvents = UNRECOVERABLE_FAIL_EVENTS.join(" ");

      function handleFailureEvent(ev, sourceWidget, err) {
        if (!widget) {
          widget = Widget.in(jq);
        }
        //TODO: widget is the main widget view. sourceWidget is the guy who actually threw the error.
        //if for example a string editor fails to load, should we show StringEditor
        //on the error screen instead of MultiSheet?
        error(jq, widget, ev.type, err);
      }

      // Listen for failure events and only handle this once.
      jq.one(failureEvents, handleFailureEvent);

      jq.on(RUNTIME_FAIL_EVENTS.join(' '), handleRecoverableError);
    }

////////////////////////////////////////////////////////////////
// Layout
////////////////////////////////////////////////////////////////

    var layoutWidgets = debounceAsync(function layoutWidgets() {
      var i,
        obj,
        promises = [];

      for (i = 0; i < registered.length; ++i) {
        obj = registered[i];

        if (obj && obj.widget && obj.widget.isInitialized()) {
          var jq = obj.widget.jq();
          if (!jq.data("lastWidth") || !jq.data("lastHeight")) {
            jq.data("lastWidth", jq.width());
            jq.data("lastHeight", jq.height());
          }
          promises.push(obj.widget.layout());
        }
      }

      return Promise.all(promises);
    }, 200);

    $(window).on('resize', function () {
      log.timing(layoutWidgets, 'FINE', 'Laid out widgets due to window resize in {}ms');
    });
    //splitpaneresize: see split-pane.js - when content view is resized
    //sidebar-toggle: see profileSideBars.js - when a sidebar is toggled
    $(document).on('splitpaneresize sidebar-toggle', '.WebShell-sidebar-left', function () {
      log.timing(layoutWidgets, 'FINE', 'Laid out widgets due to sidebar resize in {}ms');
    });

////////////////////////////////////////////////////////////////
// Save
////////////////////////////////////////////////////////////////

    /**
     * Save the registered widgets, or just the particular Widget if its passed in
     * @param {bajaux.Widget} [widget]
     * @returns {Promise}
     */
    function save(widget) {
      var ws,
        i;

      if (widget) {
        for (i = 0; i < registered.length; ++i) {
          if (registered[i].widget === widget) {
            ws = [ registered[i] ];
          }
        }
      } else {
        ws = registered.slice(0);
      }

      if (!ws) {
        return Promise.reject(new Error(hxLex.get('cannotSaveWidgetIsNotRegistered'))); //Widget to save is not registered
      }

      return Promise.map(ws, function (obj) {
          if (!obj.widget.isDestroyed()) {
            isSavePending = true;
            return obj.widget.save();
          }
        })
        .catch(function (err) {
          baja.error(err);
          updateSaveErrorMessage(err instanceof Error ? err.message : err);
          throw err; //ensure that any callers to save don't consider that it was successful
        }).finally(function () {
          if (saveErrorMessage) {
            showErrorDialog(saveErrorMessage);
          }
          saveErrorMessage = null;
          isSavePending = false;
        });
    }

    function isModified() {
      var modified = false;
      $.each(registered, function (i, obj) {
        if (obj.widget.isModified() && !obj.widget.isDestroyed()) {
          modified = true;
          return false;
        }
      });
      return modified;
    }

    window.onbeforeunload = function () {
      if (isModified()) {
        return hxLex.get("exitBeforeSave");
      }
    };

    var originalCloseDialog = hx.closeDialog;
    hx.$closeDialog = originalCloseDialog;

    /**
     * When hx dialogs are closed, destroy any bajaux widgets afterwards
     */
    window.hx.closeDialog = function (path, eventId, event) {
      var args = arguments,
        dlg = hx.$(hx.dialogId + hx.dialogCounter);

      var widgets = $(dlg).find('.bajaux-initialized')
        .map(function () {
          return $(this).data('widget');
        })
        .get();

      if (!widgets.length) {
        originalCloseDialog.apply(hx, args);
        return Promise.resolve();
      }

      //since we are encoding form values when path and eventId are non-null,show error dialog for any
      //bajaux editor that cannot validate
      return Promise.resolve().then(function () {
        if (path !== null && eventId !== null) {
          return Promise.map(widgets, function (widget) {
            return widget.validate();
          })
            .catch(function (err) {
              baja.error(err);

              dialogs.showOk({
                title: hxLex.get('cannotSave'),
                content: escapeHtml(err instanceof Error ? err.message : err)
              });

              throw err; //don't close the dialog if OK is pressed and all field editors cannot be validated
            });
        }
      }).then(function () {
        //close dialog
        try {
          originalCloseDialog.apply(hx, args);
        } finally {
          return destroyWidgets(widgets);
        }
      });
    };

    var originalSave = hx.save;

    /**
     * When hx save is called, ensure all ux editors saved before getting hx to complete save
     */
    window.hx.save = function () {
      var args = arguments;
      return save().then(function () {
        originalSave.apply(hx, args);
      });
    };

    ////////////////////////////////////////////////////////////////
    // Env
    ////////////////////////////////////////////////////////////////

    window.niagara = window.niagara || {};
    window.niagara.env = window.niagara.env || {};

    /**
     * Starting in Niagara 4.10U7, this function will also contact the station
     * for the proper normalization of any `baja:ISubstitutableOrdScheme`.
     * This additional behavior can be skipped with providing an additional parameter `$skipStationSideNormalization`.
     *
     * @param {string|baja.Ord} ord
     * @param {Object} [params] - optional parameter object.
     * @param {boolean} [params.$skipStationSideNormalization=false] If set to true, skips the station side normalization.
     * @returns {Promise.<string>}
     */
    window.niagara.env.toHyperlink = function toHyperlink(ord, params) {
      if (!params) {
        params = {};
      }
      
      var uri = String(ord);
      //external urls like http://www.example.com do not need normalization or need contact with the view all ord servlet
      if (uri.startsWith('http')) {
        return Promise.resolve(uri);
      }
      return Promise.try(function () {
        return getNormalizedUri(uri);
      })
        .then(function (result) {
          uri = result;
          if (params.$skipStationSideNormalization) {
            return uri;
          }
          return log.timing(function () {
            return Promise.resolve($.ajax("/view/all" + uri, {
              type: "GET",
              contentType: "application/json",
              dataType: "json"
            }));
          }, 'FINE', 'retrieved view info for hyperlinking to {} in {}ms', uri)
            .then(function (viewInfo) {
              viewInfo = viewInfo ||  [];
              var  list = viewInfo.list;
              var def = list.find(function (item) {
                return item.def;
              });
              if (def) {
                var defUri = def.uri;
                var defOrd = uriToOrd(defUri);
                var viewQuery = getViewQuery(uriToOrd(uri));
                if (!viewQuery) {
                  return defUri;
                }
                var ordToLoad = baja.Ord.make({
                  base: defOrd,
                  child: viewQuery
                }).normalize();
                return ordToLoad.toUri();
              } else {
                return uri;
              }
            });
        })
        .catch(function (err) {

          try {
            //if this uri can be made into an ord, then ensure it uses the toUri form
            uri = baja.Ord.make(uri).toUri();
          } catch (ignore) {
          }
          
          logInfo("toHyperlink could not resolve ord in station, falling back to original ord: " + uri, err);
          return uri;
        });
    };

    window.niagara.env.hyperlink = function hyperlink(ordStr, assign) {
      var pWin = baja.parentWindow();

      // This hyperlink call should look up to the top-most parent window to process
      // the hyperlink, but if none can be found, go ahead and process it.
      if (!pWin || pWin === window || typeof pWin.niagara === 'undefined') {
        return window.niagara.env.toHyperlink(ordStr)
          .then(function (uri) {
            if (typeof assign === "function") {
              assign(uri);
            } else {
              window.location.assign(uri);
            }
          }).catch(function (err) {
            logError(err);
            var uri = baja.Ord.make(ordStr).toUri();
            if (typeof assign === "function") {
              assign(uri);
            } else {
              window.location.assign(uri);
            }
          });
      }

      // Walk up to the top-most parent window that can process the hyperlink
      var potentialParent = baja.parentWindow(pWin);
      while (potentialParent && potentialParent !== pWin && typeof potentialParent.niagara !== 'undefined') {
        pWin = potentialParent;
      }

      return pWin.niagara.env.hyperlink(ordStr, assign);
    };

    window.niagara.env.reload = function reload() {
      return window.niagara.env.hyperlink(profileUtils.getCurrentOrd());
    };

    /**
     * Start a drag operation in Hx world. The data will be encoded onto the
     * event's `dataTransfer` clipboard.
     *
     * @param {String|Object} json a JSON encoded string, or an object to
     * be encoded to JSON. The encoded object should have a String `mime`
     * property, and an Array `data` property.
     * @param {jQuery.Event} ev the jQuery dragstart event
     */
    window.niagara.env.startDrag = function startDrag(json, ev) {
      ev = ev.originalEvent || ev;

      if (typeof json === 'string') {
        json = JSON.parse(json);
      }

      if (!json.mime || !json.data) {
        throw new Error('json object should have "mime" and "data" properties');
      }

      dragDropUtils.toClipboard(ev.dataTransfer, json.mime, json.data).catch(baja.error);
    };
////////////////////////////////////////////////////////////////
// Hyperlink
////////////////////////////////////////////////////////////////

    /**
     * Utility to get a uri from the passed in ord string
     *  
     * @param {string} fromOrdStr 
     * @returns {string} a valid and normalized uri
     */
    function getNormalizedUri(fromOrdStr) {
      if (fromOrdStr.charAt(0) === "/") {
        return fromOrdStr;
      } else {
        if (fromOrdStr.match(/^file:/)) {
          return baja.Ord.make(fromOrdStr).toUri();
        } else {
          return baja.Ord.make({
            base: profileUtils.getCurrentOrd().relativizeToSession(),
            child: fromOrdStr
          }).toUri();
        }
      }
    }

////////////////////////////////////////////////////////////////
// Properties
////////////////////////////////////////////////////////////////

    /**
     * Make a post request to the station.
     * @param {Object} json
     * @param {String} contentType
     * @returns {Promise}
     */
    function ajax(json, contentType) {
      // Make an Hx process call to the HxWebWidget.
      return Promise.resolve($.ajax(window.location.href, {
        type: "POST",
        headers: {
          'x-niagara-csrfToken': hx.getCsrfToken()
        },
        contentType: contentType,
        dataType: "json",
        data: JSON.stringify(json),
        processData: false
      }));
    }

    function updateProperty(properties, propName, updatedProp) {
      var existingProp = properties.get(propName),
        valueDefined = updatedProp.value !== undefined,
        metadataDefined = updatedProp.metadata !== undefined;
      if (existingProp && !existingProp.userModified) {

        if (valueDefined || metadataDefined) {

          if (valueDefined) {
            properties.setValue(propName, updatedProp.value, setValueParam);
          }

          if (metadataDefined) {
            properties.setMetadata(propName, updatedProp.metadata, setMetadataParam);
          }
        } else {
          properties.setValue(propName, updatedProp, setValueParam);
        }
      } else if (!existingProp && propName === "enabled") {
        properties
          .add({
            name: propName,
            value: valueDefined ? updatedProp.value : updatedProp
          });

        if (metadataDefined) {
          properties.setMetadata(propName, updatedProp.metadata, setMetadataParam);
        }
      }
    }

    function syncProperties(widget, id) {
      var props = widget.properties();
      if (props.size()) {
        ajax({
          id: id,
          props: props.get()
        }, "application/x-niagara-hx-bux-sync-props")
          .then(function (updatedProps) {
            // If we get any updates back then process them.
            if (updatedProps) {
              Object.keys(updatedProps).forEach(function (propName) {
                updateProperty(props, propName, updatedProps[propName]);
              });
            }
          }).catch(function (err) {
          console.error("Could not sync props: " + err);
        });
      }
    }

    function sendJsPropToServer(widget, id, name, value) {
      ajax({ id: id, name: name, value: value },
        "application/x-niagara-hx-bux-prop-change")
        .then(function (prop) {
          if (prop) {
            widget.properties()
              .setValue(prop.name, prop.value, setValueParam);
          }
        }).catch(baja.error);
    }

    function sendJsPropMetadataToServer(widget, id, name, metadata) {
      ajax({ id: id, name: name, metadata: metadata },
        "application/x-niagara-hx-bux-meta-change")
        .then(function (prop) {
          if (prop) {
            widget.properties()
              .setMetadata(prop.name, prop.metadata, setValueParam);
          }
        }).catch(baja.error);
    }

////////////////////////////////////////////////////////////////
// Destroy
////////////////////////////////////////////////////////////////

    function cleanup(containerJq, jq, toolbarJq, id) {
      containerJq.off();
      jq.off();
      toolbarJq.off();

      var i,
        w,
        destroyTimeId,
        destroyPromise;

      for (i = 0; i < registered.length; ++i) {
        if (id === registered[i].id) {
          w = registered[i].widget;
          registered.splice(i, 1);

          if (canProfileBrowser() && console.time) {
            destroyTimeId = "Destroy: " + id;
            console.time(destroyTimeId);
          }

          util.preDestroy(w);
          destroyPromise = w.destroy();
          break;
        }
      }
      return Promise.resolve(destroyPromise).then(function () {
        if (destroyTimeId && console.timeEnd) {
          console.timeEnd(destroyTimeId);
        }
      });
    }

////////////////////////////////////////////////////////////////
// Initialize
////////////////////////////////////////////////////////////////

    function hxCommFail() {
      dialogs.show({
        title: bajauxLex.get("error.title"),
        content: hxLex.getSafe("commsFail")
      });
    }

    function setupCommsFailure() {
      baja.comm.setCommFailCallback(hxCommFail);
    }

    function initialize(widget, id, props, parameters) {
      // Hx adds illegal 'dots' so get the DOM element via
      // document.getElementbyId and then wrap in jQuery.
      var containerJq = $(document.getElementById(id)),
        jq = containerJq.children(".bajaux-widget-container").children(".bajaux-widget"),
        toolbarJq = containerJq.children(".bajaux-toolbar-outer").children(".bajaux-toolbar"),
        properties = widget.properties(),
        timeId,
        skipCommandBar = parameters && parameters.previewMedia === "true";

      if (props) {
        jq.data("lastProps", JSON.stringify(props));
      }

      function propsFromParameters() {
        if (parameters) {
          $.each(parameters, function (name, value) {
            value = util.decodeValueFromParameter(properties, name, value);
            if (value !== null) {
              properties.setValue(name, value, setValueParam);
            }
          });
        }
      }

      function getPropChanges() {
        return props;
      }

      function propChanges(changes) {
        if (changes) {
          Object.keys(changes).forEach(function (name) {
            updateProperty(properties, name, changes[name]);
          });
        }
      }

      /**
       * Configure the enabled state based off the initial value of the enabled property.
       * @return {Promise}
       */
      function configureEnabled() {
        if (properties.getValue("enabled") === false && widget.isEnabled()) {
          return widget.setEnabled(false);
        }
        return Promise.resolve();
      }


      function init() {
        if (canProfileBrowser() && console.time) {
          timeId = "Init: " + id;
          console.time(timeId);
        }

        // Initialize the widget with the given DOM element
        return widget.initialize(jq);
      }

      function initCommands() {
        if (!skipCommandBar) {
          return util.initCommandBar(widget, containerJq);
        }
      }

      // Arm and Register the widget
      function register() {
        var hxPx = window &&
          window.niagara &&
          window.niagara.env &&
          window.niagara.env.type === "hxPx";

        //TODO: not a memory leak per se, but makes diagnosing memory leaks harder than it has to be.
        //provide some external way to clean up this array.
        registered.push({
          widget: widget,
          id: id
        });

        // When Properties are added and removed, update the Hx widget in the station.
        jq.on(events.PROPERTY_ADDED + " " + events.PROPERTY_REMOVED, function (ev, w, name) {

          if (w !== widget) {
            return;
          }

          if (hxPx) {
            syncProperties(widget, id);
          }

          if (name === hideCommandBar) {
            util.toggleCommandBar(widget, toolbarJq);
          }
        });

        // When a Property is changed, update the Hx widget in the station.
        jq.on(events.PROPERTY_CHANGED, function (ev, w, name, value, params) {
          if (w !== widget) {
            return;
          }

          if (hxPx && params !== setValueParam) {
            sendJsPropToServer(widget, id, name, value);
          }
          if (name === hideCommandBar) {
            util.toggleCommandBar(widget, toolbarJq);
          }
        });

        // When a Property's meta data is changed, update the Hx widget in the station.
        jq.on(events.METADATA_CHANGED, function (ev, w, name, metadata, params) {
          if (w !== widget) {
            return;
          }

          if (hxPx && params !== setValueParam) {
            sendJsPropMetadataToServer(widget, id, name, metadata);
          }
          if (name === hideCommandBar) {
            util.toggleCommandBar(widget, toolbarJq);
          }
        });

        // When the Command Group changes, reinitialize the Command Bar.
        jq.on(events.command.GROUP_CHANGE_EVENT, initCommands);

        if (hxPx) {
          // Add a function to the container DOM object. This can
          // be easily called from the hx update method.
          containerJq.data("hxBajauxUpdate", function (propsToChange, width, height) {


            var lastProps = jq.data("lastProps"),
              currentProps = propsToChange ? JSON.stringify(propsToChange) : null,
              propsHaveChanged = currentProps !== lastProps;

            jq.data("lastProps", currentProps);

            if (propsHaveChanged) {
              propChanges.apply(this, arguments);
            }

            //change the layouts of all widgets when either the props have changed or the dimensions
            // for a widget have changed.
            if (propsHaveChanged || checkForDimensionChange(jq, width, height)) {

              layoutWidgets();
            }
          });

          // Synchronize the Properties with the Server.
          syncProperties(widget, id);
        }

        if (timeId && console.timeEnd) {
          console.timeEnd(timeId);
        }

        return widget;
      }

      return Promise.resolve(setupCommsFailure())
        .then(function () {
          return cleanup(containerJq, jq, toolbarJq, id);
        })
        .then(function () {
          return armFailureHandlers(jq, widget);
        })
        .then(propsFromParameters)
        .then(getPropChanges)
        .then(propChanges)
        .then(configureEnabled)
        .then(init)
        .then(initCommands)
        .then(register)
        .catch(function (err) {
          errorHandler(jq, widget, "initializing", err);
        });
    }

    /**
     * Check to see if the dimensions have changed since the last call to layout
     * @param {JQuery} jq
     * @param {Number} width
     * @param {Number} height
     * @returns {boolean}
     */
    function checkForDimensionChange(jq, width, height) {
      if (width === undefined || height === undefined) {
        return false;
      }

      var needsLayout = jq.data("lastWidth") !== width || jq.data("lastHeight") !== height;
      if (needsLayout) {
        jq.data("lastWidth", width);
        jq.data("lastHeight", height);
      }
      return needsLayout;
    }

////////////////////////////////////////////////////////////////
// Load
//////////////////////////////////////////////////////////////// 

    function load(widget, ord) {
      var timeId;

      if (canProfileBrowser() && console.time) {
        timeId = "Load: " + widget.jq().parent().parent().attr("id") + " (" + ord + ")";
        console.time(timeId);
      }

      function resolve() {
        // Initialize the widget with the given DOM element
        return Promise.resolve(widget.resolve(ord));
      }

      // Load the value into the widget
      function loadValue(value) {
        return widget.load(value);
      }

      function layout() {
        return widget.layout();
      }

      return resolve()
        .then(loadValue)
        .then(layout)
        .then(function () {
          if (timeId && console.timeEnd) {
            console.timeEnd(timeId);
          }
        })
        .catch(function (err) {
          errorHandler(widget.jq(), widget, "loading", err);
        });
    }

    /**
     * Update the form for hx when the Bajaux editor has been modified.
     * @param {Widget} widget
     * @param {String} id
     */
    function updateFormOnModify(widget, id) {
      widget.jq().on(events.MODIFY_EVENT, function (event) {
        return widget.validate()
          .then(function (diff) {
            if (diff instanceof ComplexDiff) {
              var value = widget.value().newCopy(true);
              return diff.apply(value);
            }
            return diff;
          }).then(function (value) {
            var valueId = id + ".value";
            var dialog = widget.jq().closest(".dialog");
            var inputValue = JSON.stringify(baja.bson.encodeValue(value));
            if (dialog.length > 0) {
              var input = dialog.find("#" + hx.escapeSelector(valueId));
              var control;
              if (input.length === 0) {
                control = document.createElement("input");
                control.type = "hidden";
                control.name = valueId;
                control.id = valueId;
              } else {
                control = input[0];
              }
              control.value = inputValue;
              dialog[0].appendChild(control);
            } else {
              hx.setFormValue(valueId, inputValue);
              hx.modified(event.target);
            }
          }).catch(function (err) {
            baja.error(err);
          });
      });
    }

    /**
     * buildFor creates a field editor for the desired slot on a component. For bound values, you can provide the
     * complexOrd and slot. For unbound values, you can just provide the bson encoding of the value.
     *
     * @param {baja.Ord} [complexOrd]
     * @param {String} [slot]
     * @param {String} [valueEncoding] Only used when fieldEditor is not bound
     * @param {String} id
     * @param {Object} [facetsEncoding]
     * @param {Boolean} [readonly=false]
     * @param {Boolean} [enabled=true]
     */
    function buildFor(complexOrd, slot, valueEncoding, id, facetsEncoding, readonly, enabled) {

      var containerJq = $(document.getElementById(id)),
        widgetContainerJq = containerJq.children(".bajaux-widget-container"),
        toolbarJq = containerJq.children(".bajaux-toolbar-outer").children(".bajaux-toolbar"),
        facetsPromise,
        timeId,
        widget,
        jq = $('<span></span>'),
        valuePromise;

      widgetContainerJq.append(jq);

      enabled = enabled !== undefined ? enabled : true;

      if (valueEncoding) {
        valuePromise = baja.bson.decodeAsync(JSON.parse(valueEncoding));
      }

      if (facetsEncoding) {
        facetsPromise = baja.bson.decodeAsync(JSON.parse(facetsEncoding));
      }

      function init() {
        if (canProfileBrowser() && console.time) {
          timeId = "Init: " + id;
          console.time(timeId);
        }

        var value;
        // Initialize the widget with the given DOM element
        return (valuePromise || baja.Ord.make('station:|' + complexOrd).get({ lease: true }))
          .then(function (valueResult) {
            value = valueResult;
            return facetsPromise;
          }).then(function (facets) {


            var buildForParams = {
              formFactor: 'mini'
            };

            if (facets) {
              buildForParams.properties = facetsUtils.toProperties(facets);
            }

            var cssClass = 'hx-bajaux-editor';

            jq.addClass(cssClass);

            buildForParams.dom = jq;

            if (readonly) {
              buildForParams.readonly = true;
            }

            if (!enabled) {
              buildForParams.enabled = false;
            }

            if (slot && value.getType().isComplex()) {
              buildForParams.slot = slot;
              buildForParams.complex = value;
            } else {
              buildForParams.value = value;
            }

            //Passing along the complexOrd to the field editor load helps
            //certain custom field editors subscribe properly like HxSetPointFE.
            buildForParams.loadParams = { complexOrd: complexOrd, slot: slot };

            return fe.buildFor(buildForParams)
              .then(function (editor) {
                widget = editor;
                updateFormOnModify(widget, id);
                //no double popup required for when the PropertySheet is used
                if (buildForParams.properties &&
                  buildForParams.properties.getValue("uxFieldEditor") === "nmodule/webEditors/rc/wb/PropertySheet") {
                  return;
                }
                var commandJq = $('<span class="hx-popout-command"></span>');
                jq.append(commandJq);

                var commandGroup = new CommandGroup({
                  displayName: 'inline',
                  jq: commandJq,
                  commands: [
                    new PopOutCommand(editor)
                  ]
                });

                return fe.buildFor(
                  {
                    dom: commandJq,
                    value: commandGroup,
                    type: CommandButtonGroup,
                    formFactor: 'mini',
                    properties: { toolbar: true, onDisabled: 'hide' }
                  });
              });
          });
      }

      // Arm and Register the widget
      function registerWithBuildFor() {
        //TODO: not a memory leak per se, but makes diagnosing memory leaks harder than it has to be.
        //provide some external way to clean up this array.
        registered.push({
          widget: widget,
          id: id
        });

        if (timeId && console.timeEnd) {
          console.timeEnd(timeId);
        }

        return widget;
      }

      function armUpdate() {
        // Add a function to the container DOM object. This can
        // be easily called from the hx update method.
        containerJq.data("hxBajauxUpdate", function (propsToChange) {

          var enabled = true;

          if (propsToChange && propsToChange.enabled && !propsToChange.enabled.value) {
            enabled = false;
          }

          if (widget.isEnabled() !== enabled) {
            widget.setEnabled(enabled);
          }
        });
      }

      return Promise.resolve(setupCommsFailure())
        .then(function () {
          return cleanup(containerJq, jq, toolbarJq, id);
        })
        .then(function () {
          return armFailureHandlers(jq, widget);
        })
        .then(init)
        .then(registerWithBuildFor)
        .then(armUpdate)
        .catch(function (err) {
          errorHandler(jq, widget, "initializing", err);
        });
    }

////////////////////////////////////////////////////////////////
// Export
//////////////////////////////////////////////////////////////// 

    exports = {
      initialize: initialize,
      load: load,
      save: save,
      buildFor: buildFor,
      isModified: isModified,
      isError: isError,
      error: function (id, type, err) {
        var containerJq = $(document.getElementById(id)),
          jq = containerJq.children(".bajaux-widget-container").children(".bajaux-widget");

        error(jq, Widget.in(jq), type, err);
      },

      //export this function for testing
      $getNormalizedUri: getNormalizedUri
    };

    define("bajauxContainer", exports);

    return exports;
  }
)
;
