/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/hierarchy/rc/fe/RoleHierarchiesCompactEditor
 */
define(['baja!',
        'baja!hierarchy:RoleHierarchies,hierarchy:HierarchyService,hierarchy:Hierarchy',
        'bajaux/events',
        'nmodule/webEditors/rc/fe/baja/BaseEditor',
        'bajaux/mixin/subscriberMixIn',
        'nmodule/webEditors/rc/fe/fe',
        'nmodule/webEditors/rc/fe/baja/util/spaceUtils',
        'nmodule/webEditors/rc/fe/baja/util/subscriberUtils',
        'jquery',
        'underscore',
        'Promise',
        'css!nmodule/hierarchy/rc/hierarchy',
       ], function (
         baja,
         types,
         events,
         BaseEditor,
         subscriberMixIn,
         fe,
         spaceUtils,
         subscriberUtils,
         $,
         _,
         Promise
       ) {

  'use strict';

  var MODIFY_EVENT = events.MODIFY_EVENT,
    NAMES_DELIMITER = ',',
    resolveService = spaceUtils.resolveService,
    leaseWhenNeeded = subscriberUtils.leaseWhenNeeded,
    getHierarchyService = leaseWhenNeeded(_.partial(resolveService, 'hierarchy:HierarchyService'));


  /**
   * A compact field editor for working with RoleHierarchies
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/RoleHierarchiesCompactEditor
   */
  var RoleHierarchiesCompactEditor = function RoleHierarchiesCompactEditor(params) {
    var that = this;

    BaseEditor.call(that, $.extend(true, {
      keyName: 'RoleHierarchiesCompactEditor'
    }, params));

    subscriberMixIn(that);
  };

  RoleHierarchiesCompactEditor.prototype = Object.create(BaseEditor.prototype);
  RoleHierarchiesCompactEditor.prototype.constructor = RoleHierarchiesCompactEditor;

  /**
   * Creates a RoleHierarchiesCompactEditor element.
   *
   * @param {JQuery} dom
   */
  RoleHierarchiesCompactEditor.prototype.doInitialize = function (dom) {
    var that = this;

    //if a row is modified, set myself modified
    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });
  };

  /**
   * Loads the given RoleHierachies component into the editor.
   *
   * @param {baja.Component} roleHierarchies - a hierarchy:RoleHierarchies component
   * @returns {Promise}
   */
  RoleHierarchiesCompactEditor.prototype.doLoad = function (roleHierarchies) {
    var that = this,
      dom = that.jq(),
      isOnAdminRole = that.$isOnAdminRole();

    return that.$getHierarchies().then(function (hierarchies) {
      var hierarchyNames = roleHierarchies.getHierarchyNames().split(',');
      return Promise.all(
        hierarchies.map(function (hierarchy) {

          return fe.buildFor({
            dom:  $('<div class="roleHierarchyEditor" data-hierarchy-name="' + hierarchy.getName() + '"/>').appendTo(dom),
            value: isOnAdminRole || hierarchyNames.indexOf( hierarchy.getName() ) >= 0,
            properties: {
              trueText: hierarchy.$getDisplayName(),
              falseText: hierarchy.$getDisplayName()
            },
            formFactor: 'mini',
            enabled: !isOnAdminRole
          });
        })
      );
    });
  };

  /**
   * Read the values from the editor
   *
   * @returns {Promise.<baja.Component>} promise to be resolved with a a hierarchy:RoleHierarchies component
   */
  RoleHierarchiesCompactEditor.prototype.doRead = function () {
    var that = this,
      selectedHierarchyNames = [];

    // don't apply any changes on the Admin Role, return the the (unchanged) underlying value
    if (that.$isOnAdminRole()) {  return Promise.resolve(that.value()); }

    return Promise.all(
      that.getChildEditors().map(function (editor) {
        return editor.read().then(function (val) {
          if (val) { selectedHierarchyNames.push(editor.jq().data('hierarchy-name')); }
        });
      })
    )
    .then(function () {
      return baja.$('hierarchy:RoleHierarchies',{
        hierarchyNames: selectedHierarchyNames.join(NAMES_DELIMITER)
      });
    });
  };

  /**
   * save the RoleHierachies component
   * @param {baja.Component} readValue - a hierarchy:RoleHierarchies component
   * @param {Object} params
   */
  RoleHierarchiesCompactEditor.prototype.doSave = function ( readValue, params) {
    var that = this;

    // don't attempt any changes on the Admin Role
    if (that.$isOnAdminRole()) { return; }

    return that.value().setHierarchyNames(readValue.getHierarchyNames());
  };

  RoleHierarchiesCompactEditor.prototype.$getHierarchyService = function () {
    return this.getOrdBase().then(getHierarchyService);
  };

  RoleHierarchiesCompactEditor.prototype.$getHierarchies = function () {
    return this.$getHierarchyService()
      .then(function (service) {
        return service.getSlots().properties().is('hierarchy:Hierarchy')
          // In BRoleHierarchiesFE, there's a TODO related to Licensing that may use hierarchySelection.getStatus().isOk() in the future.
          // The equivalent here would be something like this:
          // .filter(function (hierarchy) {
          //   return hierarchy.$getValue().getStatus().isOk();
          // })
          .toArray();
      })
      .catch(_.constant(null));
  };

  RoleHierarchiesCompactEditor.prototype.$isOnAdminRole = function () {
    return baja.hasType(this.value().getParent(), 'baja:AdminRole');
  };

  /**
   * Enables or disables the view of the RoleHierarchiesCompactEditor.
   *
   * @param {Promise} enabled
   */
  RoleHierarchiesCompactEditor.prototype.doEnabled = function (enabled) {
    return this.getChildEditors().setAllEnabled(enabled);
  };

  /**
   * Make the RoleHierarchiesCompactEditor readonly (or not).
   *
   * @param {Promise} readonly
   */
  RoleHierarchiesCompactEditor.prototype.doReadonly = function (readonly) {
    return this.getChildEditors().setAllReadonly(readonly);
   };

  /**
   * Destroys all child editors.
   *
   * @returns {Promise} promise to be resolved when all child editors
   * have been destroyed
   */
  RoleHierarchiesCompactEditor.prototype.doDestroy = function () {
    return this.getChildEditors().destroyAll();
  };

  return RoleHierarchiesCompactEditor;
});
